/*
 * FragmentNode.hpp
 *
 *  Created on: Sep 24, 2012
 *      Author: heber
 */

#ifndef FRAGMENTNODE_HPP_
#define FRAGMENTNODE_HPP_

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <boost/serialization/export.hpp>

#include <iosfwd>

/** FragmentNode contains all information important to homology considerations
 * between fragments.
 *
 * This is important to answer whether to fragments have the same structure or
 * not. Basically, the node simply represents an atom but without any index to
 * make it indistinguishable from another atom having same number of bonds and
 * same atomic number.
 *
 * Namely, we need:
 * -# charge/atomic number of the node
 * -# number of edges connected to this node
 *
 * Central is have this class be sortable by having comparison operators defined.
 */
class FragmentNode
{
  //!> grant output operator access to internals
  friend std::ostream& operator<<(std::ostream &out, const FragmentNode &node);
public:
  /** Default constructor for class FragmentNode.
   *
   * Is required to allow placement in STL containers.
   */
  FragmentNode() :
    AtomicNumber(0),
    ConnectedEdges(0)
  {}

  /** Constructor for class FragmentNode.
   *
   * @param _AtomicNumber atomic number of represented atom
   * @param _ConnectedEdges number of "bonds" of represented atom
   */
  FragmentNode(const size_t _AtomicNumber, const size_t _ConnectedEdges) :
    AtomicNumber(_AtomicNumber),
    ConnectedEdges(_ConnectedEdges)
  {}
  ~FragmentNode()
  {}

  FragmentNode& operator=(const FragmentNode &node);

  // comparison operators to allow for sorting
  bool operator<(const FragmentNode &node) const;
  bool operator>(const FragmentNode &node) const;
  bool operator==(const FragmentNode &node) const;
  bool operator!=(const FragmentNode &node) const {
    return (!(*this == node));
  }

private:
  //!> the atomic number
  const size_t AtomicNumber;
  //!> number of connecting edges
  const size_t ConnectedEdges;

private:
  friend class boost::serialization::access;
  // serialization
  template <typename Archive>
  void serialize(Archive& ar, const unsigned int version)
  {
    ar & const_cast<size_t &>(AtomicNumber);
    ar & const_cast<size_t &>(ConnectedEdges);
  }
};

// we need to give this class a unique key for serialization
BOOST_CLASS_EXPORT_KEY(FragmentNode)

std::ostream& operator<<(std::ostream &out, const FragmentNode &node);

#endif /* FRAGMENTNODE_HPP_ */
