/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2012 University of Bonn. All rights reserved.
 * Please see the COPYING file or "Copyright notice" in builder.cpp for details.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>. 
 */

/*
 * FragmentNode.cpp
 *
 *  Created on: Sep 24, 2012
 *      Author: heber
 */


// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

// include headers that implement a archive in simple text format
// otherwise BOOST_CLASS_EXPORT_IMPLEMENT has no effect
#include <boost/archive/text_oarchive.hpp>
#include <boost/archive/text_iarchive.hpp>

#include "CodePatterns/MemDebug.hpp"

#include "FragmentNode.hpp"

#include <iostream>

/** Assignment operator for class FragmentNode.
 *
 * We need this assignment to overcome the constness of the member. For STL
 * containers default cstors and usable assignment ops are required. This
 * const_cast is only way to allow for const member variables.
 *
 * @param edge other instance to use in assignment
 * @return this instance for concatenation
 */
FragmentNode& FragmentNode::operator=(const FragmentNode &node)
{
  // check self-assignment
  if (this != &node) {
    const_cast<size_t &>(this->AtomicNumber) = node.AtomicNumber;
    const_cast<size_t &>(this->ConnectedEdges) = node.ConnectedEdges;
  }
  return *this;
}

bool FragmentNode::operator<(const FragmentNode &node) const
{
  if (AtomicNumber < node.AtomicNumber) {
    return true;
  } else if (AtomicNumber > node.AtomicNumber) {
    return false;
  } else {
    if (ConnectedEdges < node.ConnectedEdges)
      return true;
    else
      return false;
  }
}

bool FragmentNode::operator>(const FragmentNode &node) const
{
  if (AtomicNumber > node.AtomicNumber) {
    return true;
  } else if (AtomicNumber < node.AtomicNumber) {
    return false;
  } else {
    if (ConnectedEdges > node.ConnectedEdges)
      return true;
    else
      return false;
  }
}

bool FragmentNode::operator==(const FragmentNode &node) const
{
  if (AtomicNumber != node.AtomicNumber) {
    return false;
  } else {
    return (ConnectedEdges == node.ConnectedEdges);
  }
}

std::ostream& operator<<(std::ostream &out, const FragmentNode &node)
{
  out << "[Z" << node.AtomicNumber << ",E" << node.ConnectedEdges << "]";
  return out;
}


// we need to explicitly instantiate the serialization functions
BOOST_CLASS_EXPORT_IMPLEMENT(FragmentNode)
