/*
 * Extractors.hpp
 *
 *  Created on: 15.10.2012
 *      Author: heber
 */

#ifndef TRAININGDATA_EXTRACTORS_HPP_
#define TRAININGDATA_EXTRACTORS_HPP_

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <boost/function.hpp>

#include "Fragmentation/Summation/SetValues/Fragment.hpp"
#include "FunctionApproximation/FunctionModel.hpp"

class Fragment;
class HomologyGraph;

/** Namespace containing all simple extractor functions.
 *
 * Extractor functions extract distances from a given fragment matching with
 * a given set of particle types (i.e. elements, e.h. H2O).
 * Filter functions extract a subset of distances from a given set of distances
 * to be used with a specific model.
 *
 * To this end, each FunctionModel has both a filter and an extractor function.
 *
 * The functions in this namespace act as helpers or basic building blocks in
 * constructing such filters and extractors.
 *
 */
namespace Extractors {
  typedef Fragment::charges_t::const_iterator chargeiter_t;
  typedef std::vector<chargeiter_t> chargeiters_t;

  typedef size_t count_t;
  typedef Fragment::atomicNumber_t element_t;
  typedef std::map< element_t, count_t> elementcounts_t;
  typedef std::map< element_t, chargeiters_t > elementtargets_t;
  typedef std::vector< chargeiters_t > targets_per_combination_t;
  //!> typedef for particle designation
  typedef unsigned int ParticleType_t;
  //!> typedef for a vector of particle designations
  typedef std::vector<ParticleType_t> ParticleTypes_t;

  /** Namespace for some internal helper functions.
   *
   */
  namespace _detail {

    /** Counts all same elements in the vector and places into map of elements.
     *
     * \param elements vector of elements
     * \return count of same element in vector
     */
    elementcounts_t getElementCounts(
        const Fragment::atomicnumbers_t elements
        );

  }

  /** Gather all distances from a given set of positions.
   *
   *  Here, we only return one of the two equal distances.
   *
   * \param positions all nuclei positions
   * \param atomicNumber all nuclei atomic numbers
   * \param globalid index to associated in argument_t with
   * \return vector of argument_ , each with a distance
   */
  FunctionModel::arguments_t
  gatherAllSymmetricDistanceArguments(
      const Fragment::positions_t& positions,
      const Fragment::atomicnumbers_t& atomicnumbers,
      const size_t globalid);

  /** Simple extractor of all unique pair distances of a given \a fragment, where
   * the first index is less than the second one.
   *
   * \param positions all nuclei positions
   * \param atomicNumber all nuclei atomic numbers
   * \param index index refers to the index within the global set of configurations
   * \return vector of of argument_t containing all found distances
   */
  inline FunctionModel::arguments_t gatherAllSymmetricDistances(
      const Fragment::positions_t& positions,
      const Fragment::atomicnumbers_t& atomicnumbers,
      const size_t index
      ) {
    // get distance out of Fragment
    return gatherAllSymmetricDistanceArguments(positions, atomicnumbers, index);
  }

  /** Reorder the arguments to bring adjacent ones together.
   *
   * After filtering via particle types arguments related via same indices
   * must not necessarily be contained in the same bunch. This reordering
   * is done here, preserving the alignment given in
   * \sa filterArgumentsByParticleTypes()
   *
   * \param listargs list of arguments to reorder each
   * \param _types particle type vector
   * \return reordered args
   */
  FunctionModel::list_of_arguments_t reorderArgumentsByParticleTypes(
      const FunctionModel::list_of_arguments_t &eachargs,
      const ParticleTypes_t &_types,
      const HomologyGraph &_bindingmodel
      );

  /** Filter arguments according to types, allowing multiples.
   *
   * If particle types is (0,1,2) and three arguments, each with a pair of types,
   * are given, then the alignment will be: (0,1), (0,2), and (1,2).
   *
   * \param args arguments to reorder
   * \param _types particle type vector
   * \return filtered list of args
   */
  FunctionModel::list_of_arguments_t filterArgumentsByParticleTypes(
      const FunctionModel::arguments_t &args,
      const ParticleTypes_t &_types,
      const HomologyGraph &_bindingmodel
      );

  /** Combines two argument lists by sorting and making unique.
   *
   * @param firstargs first list of arguments
   * @param secondargs second list of arguments
   * @return concatenated lists
   */
  FunctionModel::arguments_t combineArguments(
      const FunctionModel::arguments_t &firstargs,
      const FunctionModel::arguments_t &secondargs);

  /** Combines two argument lists by concatenation.
   *
   * @param firstargs first list of arguments
   * @param secondargs second list of arguments
   * @return concatenated lists
   */
  FunctionModel::arguments_t concatenateArguments(
      const FunctionModel::arguments_t &firstargs,
      const FunctionModel::arguments_t &secondargs);

  /** Combines two argument lists by concatenation.
   *
   * @param firstlistargs first list of argument tuples
   * @param secondlistargs second list of argument tuples
   * @return concatenated lists
   */
  FunctionModel::list_of_arguments_t concatenateListOfArguments(
      const FunctionModel::list_of_arguments_t &firstlistargs,
      const FunctionModel::list_of_arguments_t &secondlistargs);

}; /* namespace Extractors */


#endif /* TRAININGDATA_EXTRACTORS_HPP_ */
