/*
 * RandomNumberGeneratorFactory.hpp
 *
 *  Created on: Dec 31, 2010
 *      Author: heber
 */

#ifndef RANDOMNUMBERGENERATORFACTORY_HPP_
#define RANDOMNUMBERGENERATORFACTORY_HPP_

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "CodePatterns/Singleton.hpp"

#include "CodePatterns/Clone.hpp"

#include "RandomNumberDistributionFactory.hpp"
#include "RandomNumberEngineFactory.hpp"

#include <map>

#include "unittests/RandomNumberGeneratorFactoryUnitTest.hpp"

class RandomNumberGenerator;

/** This is the abstract factory class for random number generators.
 *
 * The reason for creating the generators as this is that we would like to set
 * the generator's parameters, via some Action, and then onward only have
 * random number generator of this type using these parameters. Hence, we have
 * a singleton factory that is controlled by the Action and can then create
 * generators wherever we like in the code by having the factory create one.
 *
 */
class RandomNumberGeneratorFactory : public Singleton<RandomNumberGeneratorFactory>
{
  friend class Singleton<RandomNumberGeneratorFactory>;
  friend class RandomNumberGeneratorFactoryTest;

protected:
  RandomNumberGeneratorFactory();
  virtual ~RandomNumberGeneratorFactory();

public:
  /** Create a Generator of previously set type.
   *
   * \return random number generator instance
   */
  RandomNumberGenerator& makeRandomNumberGenerator() const;

  /** Create a Generator of desired combination of engine and distribution.
   *
   * Note that this does not affect the default setting of the engine or
   * distribution type.
   *
   * \param engine_type name of engine, give empty string for current default
   * \param distribution_type name of distribution, give empty string for
   *        current default
   * \return random number generator instance
   */
  RandomNumberGenerator& makeRandomNumberGenerator(std::string engine_type, std::string distribution_type) const;

  /** Specify the precise type of the engine to build
   *
   * @param name of engine
   */
  void setEngine(std::string engine_type);

  /** Getter for the name of the current type of engine.
   *
   * @return name of engine
   */
  const std::string &getEngineName() const;

  /** Specify the precise type of the distribution to build
   *
   * @param name of distribution
   */
  void setDistribution(std::string distribution_type);

  /** Getter for the name of the current type of distribution.
   *
   * @return name of distribution
   */
  const std::string &getDistributionName() const;

protected:
private:
  /** Creates instances of all possible (engine,distribution) combinations
   *  and stores them in \a PrototypeTable.
   */
  void FillPrototypeTable();

  typedef std::map<
      RandomNumberEngineFactory::TypeList,
      std::map<
        RandomNumberDistributionFactory::TypeList,
        RandomNumberGenerator *>
  > EngineDistributionTable;

  static RandomNumberDistributionFactory::TypeList distribution;
  static RandomNumberEngineFactory::TypeList engine;
  static EngineDistributionTable GeneratorPrototypeTable;
};

#endif /* RANDOMNUMBERGENERATORFACTORY_HPP_ */
