/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010-2012 University of Bonn. All rights reserved.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * AtomsCalculationUnitTest.cpp
 *
 *  Created on: Feb 19, 2010
 *      Author: crueger
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "AtomsCalculationUnitTest.hpp"

#include <cppunit/CompilerOutputter.h>
#include <cppunit/extensions/TestFactoryRegistry.h>
#include <cppunit/ui/text/TestRunner.h>
#include <iostream>
#include <boost/bind.hpp>

#include "Descriptors/AtomDescriptor.hpp"
#include "Descriptors/AtomIdDescriptor.hpp"
#include "Actions/AtomsCalculation.hpp"
#include "Actions/AtomsCalculation_impl.hpp"
#include "Actions/ActionQueue.hpp"
#include "Actions/ActionTrait.hpp"

#include "World.hpp"
#include "World_calculations.hpp"
#include "Atom/atom.hpp"

using namespace MoleCuilder;

#ifdef HAVE_TESTRUNNER
#include "UnitTestMain.hpp"
#endif /*HAVE_TESTRUNNER*/

// Registers the fixture into the 'registry'
CPPUNIT_TEST_SUITE_REGISTRATION( AtomsCalculationTest );

// set up and tear down
void AtomsCalculationTest::setUp(){
  World::getInstance();
  for(int i=0;i<ATOM_COUNT;++i){
    atoms[i]= World::getInstance().createAtom();
    atomIds[i]= atoms[i]->getId();
  }
}
void AtomsCalculationTest::tearDown(){
  World::purgeInstance();
  ActionQueue::purgeInstance();
}

// some helper functions
static bool hasAllIds(std::vector<atomId_t> atoms,atomId_t ids[ATOM_COUNT], std::set<atomId_t> excluded = std::set<atomId_t>()){
  for(int i=0;i<ATOM_COUNT;++i){
    atomId_t id = ids[i];
    if(!excluded.count(id)){
      std::vector<atomId_t>::iterator iter;
      bool res=false;
      for(iter=atoms.begin();iter!=atoms.end();++iter){
        res |= (*iter) == id;
      }
      if(!res) {
        cout << "Atom " << id << " missing in returned list" << endl;
        return false;
      }
    }
  }
  return true;
}

static bool hasNoDuplicates(std::vector<atomId_t> ids){
  std::set<atomId_t> found;
  std::vector<atomId_t>::iterator iter;
  for(iter=ids.begin();iter!=ids.end();++iter){
    int id = (*iter);
    if(found.count(id))
      return false;
    found.insert(id);
  }
  return true;
}

void AtomsCalculationTest::testCalculateSimple(){
  ActionTrait FooTrait("FOO");
  AtomsCalculation<atomId_t> *calc = World::getInstance().calcOnAtoms<atomId_t>(boost::bind(&atom::getId,_1),FooTrait,AllAtoms());
  std::vector<atomId_t> allIds = (*calc)();
  CPPUNIT_ASSERT(hasAllIds(allIds,atomIds));
  CPPUNIT_ASSERT(hasNoDuplicates(allIds));
}

void AtomsCalculationTest::testCalculateExcluded(){
  ActionTrait FooTrait("FOO");
  atomId_t excluded = atomIds[ATOM_COUNT/2];
  AtomsCalculation<atomId_t> *calc = World::getInstance().calcOnAtoms<atomId_t>(boost::bind(&atom::getId,_1),FooTrait,AllAtoms() && !AtomById(excluded));
  std::vector<atomId_t> allIds = (*calc)();
  std::set<atomId_t> excluded_set;
  excluded_set.insert(excluded);
  CPPUNIT_ASSERT(hasAllIds(allIds,atomIds,excluded_set));
  CPPUNIT_ASSERT(hasNoDuplicates(allIds));
  CPPUNIT_ASSERT_EQUAL((size_t)(ATOM_COUNT-1),allIds.size());
}
