/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010-2012 University of Bonn. All rights reserved.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * MenuDescription.cpp
 *
 *  Created on: Oct 26, 2010
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

//#include "CodePatterns/MemDebug.hpp"

#include <iostream>
#include <map>
#include <string>

//#include "Actions/ActionQueue.hpp"
#include "Menu/MenuDescription.hpp"

#include "CodePatterns/Singleton_impl.hpp"

MenuDescription::TextMap *MenuDescription::MenuDescriptionsMap = NULL;
MenuDescription::IterableMap *MenuDescription::MenuPositionMap = NULL;
MenuDescription::TextMap *MenuDescription::MenuNameMap = NULL;


/** Constructor of class MenuDescription.
 *
 */
MenuDescription::MenuDescription()
{
  // allocate maps
  MenuDescriptionsMap = new TextMap();
  MenuPositionMap = new IterableMap();
  MenuNameMap = new TextMap();

  // put each menu into its place, "" means top level
  MenuPositionMap->insert(std::make_pair("analysis",TopPosition("tools",1)));
  MenuPositionMap->insert(std::make_pair("atom",TopPosition("edit",2)));
  MenuPositionMap->insert(std::make_pair("bond",TopPosition("edit",3)));
  MenuPositionMap->insert(std::make_pair("command",TopPosition("",3)));
  MenuPositionMap->insert(std::make_pair("edit",TopPosition("",2)));
  MenuPositionMap->insert(std::make_pair("fill",TopPosition("tools",5)));
  MenuPositionMap->insert(std::make_pair("fragmentation",TopPosition("tools",3)));
  MenuPositionMap->insert(std::make_pair("geometry",TopPosition("edit",5)));
  MenuPositionMap->insert(std::make_pair("graph",TopPosition("tools",4)));
  MenuPositionMap->insert(std::make_pair("molecule",TopPosition("edit",4)));
  MenuPositionMap->insert(std::make_pair("potential",TopPosition("tools",7)));
  MenuPositionMap->insert(std::make_pair("parser",TopPosition("edit",6)));
  MenuPositionMap->insert(std::make_pair("selection",TopPosition("edit",1)));
  MenuPositionMap->insert(std::make_pair("tesselation",TopPosition("tools",2)));
  MenuPositionMap->insert(std::make_pair("shape",TopPosition("tools",6)));
  MenuPositionMap->insert(std::make_pair("tools",TopPosition("",4)));
  MenuPositionMap->insert(std::make_pair("world",TopPosition("",1)));

  // put menu description into each menu category
  MenuDescriptionsMap->insert(std::make_pair("analysis","Analysis (pair correlation, volume)"));
  MenuDescriptionsMap->insert(std::make_pair("atom","Edit atoms"));
  MenuDescriptionsMap->insert(std::make_pair("bond","Edit bonds"));
  MenuDescriptionsMap->insert(std::make_pair("command","Configuration"));
  MenuDescriptionsMap->insert(std::make_pair("edit","Edit"));
  MenuDescriptionsMap->insert(std::make_pair("fill","Fill"));
  MenuDescriptionsMap->insert(std::make_pair("fragmentation","Fragmentation"));
  MenuDescriptionsMap->insert(std::make_pair("geometry","Geometry"));
  MenuDescriptionsMap->insert(std::make_pair("graph","Graph"));
  MenuDescriptionsMap->insert(std::make_pair("molecule","Parse files into system"));
  MenuDescriptionsMap->insert(std::make_pair("parser","Edit molecules (load, parse, save)"));
  MenuDescriptionsMap->insert(std::make_pair("potential","Fit potentials and parse, save homologies"));
  MenuDescriptionsMap->insert(std::make_pair("selection","Select atoms/molecules"));
  MenuDescriptionsMap->insert(std::make_pair("tesselation","Tesselate molecules"));
  MenuDescriptionsMap->insert(std::make_pair("shape","Edit shapes"));
  MenuDescriptionsMap->insert(std::make_pair("tools","Various tools"));
  MenuDescriptionsMap->insert(std::make_pair("world","Edit world"));

  // put menu name into each menu category
  MenuNameMap->insert(std::make_pair("analysis","Analysis"));
  MenuNameMap->insert(std::make_pair("atom","Atoms"));
  MenuNameMap->insert(std::make_pair("bond","Bonds"));
  MenuNameMap->insert(std::make_pair("command","Configuration"));
  MenuNameMap->insert(std::make_pair("edit","Edit"));
  MenuNameMap->insert(std::make_pair("fill","Fill"));
  MenuNameMap->insert(std::make_pair("fragmentation","Fragmentation"));
  MenuNameMap->insert(std::make_pair("geometry","Geometry"));
  MenuNameMap->insert(std::make_pair("graph","Graph"));
  MenuNameMap->insert(std::make_pair("molecule","Molecules"));
  MenuNameMap->insert(std::make_pair("parser","Input/Output"));
  MenuNameMap->insert(std::make_pair("potential","PotentialFitting"));
  MenuNameMap->insert(std::make_pair("selection","Selection"));
  MenuNameMap->insert(std::make_pair("tesselation","Tesselation"));
  MenuNameMap->insert(std::make_pair("shape","Shape"));
  MenuNameMap->insert(std::make_pair("tools","Tools"));
  MenuNameMap->insert(std::make_pair("world","Globals"));
}

/** Destructor of class MenuDescription.
 *
 */
MenuDescription::~MenuDescription()
{
  //std::cout << "MenuDescription: clearing maps ... " << std::endl;
  for (IterableMap::iterator iter = MenuPositionMap->begin(); !MenuPositionMap->empty(); iter = MenuPositionMap->begin())
    MenuPositionMap->erase(iter);
  delete MenuNameMap;
  delete MenuDescriptionsMap;
  delete MenuPositionMap;
}

/** Getter for MenuDescriptionsMap.
 * \param token name of menu
 * \return description string of the menu or empty
 */
const std::string MenuDescription::getDescription(const std::string &token) const
{
  if (MenuDescriptionsMap->find(token) != MenuDescriptionsMap->end())
    return MenuDescriptionsMap->find(token)->second;
  else
    return std::string();
}

/** Getter for MenuNameMap->
 * \param token name of menu
 * \return description string of the menu or empty
 */
const std::string MenuDescription::getName(const std::string &token) const
{
  if (MenuNameMap->find(token) != MenuNameMap->end())
    return MenuNameMap->find(token)->second;
  else
    return std::string();
}

///** Constructs a multimap of all menus running over all actions belonging to it.
// * \return multimap with which actions belongs to which menu.
// */
//std::multimap <std::string, std::string> MenuDescription::getMenuItemsMap() const
//{
//  std::multimap <std::string, std::string> result;
//
//  ActionQueue &AQ = ActionQueue::getInstance();
//  ActionQueue::ActionTokens_t tokens = AQ.getListOfActions();
//  for (ActionQueue::ActionTokens_t::const_iterator iter = tokens.begin();
//      iter != tokens.end(); ++iter) {
//    const ActionTrait &CurrentTrait = AQ.getActionsTrait(*iter);
//    std::cout << "Inserting " << *iter << " into menu " << CurrentTrait.getMenuName() << std::endl;
//    result.insert( std::pair<std::string, std::string> (CurrentTrait.getMenuName(), *iter));
//  }
//  // TODO: MenuPosition is not yet realized.
//  return result;
//}

/** Forward iterator from beginning of list of descriptions.
 * \return iterator
 */
MenuDescription::iterator MenuDescription::getBeginIter()
{
  return MenuPositionMap->begin();
}

/** Forward iterator at end of list of descriptions.
 * \return iterator
 */
MenuDescription::iterator MenuDescription::getEndIter()
{
  return MenuPositionMap->end();
}

/** Constant forward iterator from beginning of list of descriptions.
 * \return constant iterator
 */
MenuDescription::const_iterator MenuDescription::getBeginIter() const
{
  return MenuPositionMap->begin();
}

/** Constant forward iterator at end of list of descriptions.
 * \return constant iterator
 */
MenuDescription::const_iterator MenuDescription::getEndIter() const
{
  return MenuPositionMap->end();
}

CONSTRUCT_SINGLETON(MenuDescription)
