/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2017 Frederik Heber. All rights reserved.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * GeometryPlaneToVectorAction.cpp
 *
 *  Created on: Mar 28, 2017
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

//#include "CodePatterns/MemDebug.hpp"

#include "CodePatterns/Log.hpp"

#include <string>

#include "LinearAlgebra/Plane.hpp"

#include "Atom/atom.hpp"
#include "Geometry/GeometryObject.hpp"
#include "Geometry/GeometryRegistry.hpp"
#include "World.hpp"

#include "Actions/GeometryAction/GeometryPlaneToVectorAction.hpp"

using namespace MoleCuilder;

// and construct the stuff
#include "../GeometryAction/GeometryPlaneToVectorAction.def"
#include "Action_impl_pre.hpp"
/** =========== define the function ====================== */
ActionState::ptr GeometryPlaneToVectorAction::performCall() {
  // check preconditions
  World& world = World::getInstance();
  if (world.countSelectedAtoms() != 3) {
    STATUS("There must be exactly three atoms selected for GeometryAction PlaneToVector.");
    return Action::failure;
  }
  if (GeometryRegistry::getInstance().isPresentByName(params.name.get())) {
    STATUS("Name "+params.name.get()+" already present in registry.");
    return Action::failure;
  }

  World::AtomSelectionIterator iter = world.beginAtomSelection();
  const atom * const Walker = (iter++)->second;
  const atom * const OtherWalker = (iter++)->second;
  const atom * const AnotherWalker = (iter++)->second;
  ASSERT( iter == world.endAtomSelection(),
      "GeometryDistanceToVectorAction::performCall() - selection iter not at end?");
  Vector PlaneNormal;
  try {
    const Plane AtomPlane(Walker->getPosition(), OtherWalker->getPosition(), AnotherWalker->getPosition());
    PlaneNormal = AtomPlane.getNormal();
  } catch (LinearDependenceException &e) {
    ELOG(2, "The selected atomic positions are linearly dependent.");
    return Action::failure;
  }
  PlaneNormal *= (params.reverse.get() ? -1. : 1.);
  const GeometryObject vec(PlaneNormal, params.name.get());
  LOG(4, "DEBUG: Adding geometry object " << vec << " with norm " << PlaneNormal.Norm());
  GeometryRegistry::getInstance().addGeometry(vec);

  GeometryPlaneToVectorState *UndoState = new GeometryPlaneToVectorState(PlaneNormal, params);
  return ActionState::ptr(UndoState);
}

ActionState::ptr GeometryPlaneToVectorAction::performUndo(ActionState::ptr _state) {
//  GeometryPlaneToVectorState *state = assert_cast<GeometryPlaneToVectorState*>(_state.get());

  GeometryRegistry::getInstance().removeGeometry(params.name.get());

  return ActionState::ptr(_state);
}

ActionState::ptr GeometryPlaneToVectorAction::performRedo(ActionState::ptr _state){
  GeometryPlaneToVectorState *state = assert_cast<GeometryPlaneToVectorState*>(_state.get());

  const GeometryObject vec(state->PlaneNormal, params.name.get());
  LOG(4, "DEBUG: Adding geometry object " << vec);
  GeometryRegistry::getInstance().addGeometry(vec);

  return ActionState::ptr(_state);
}

bool GeometryPlaneToVectorAction::canUndo() {
  return true;
}

bool GeometryPlaneToVectorAction::shouldUndo() {
  return true;
}
/** =========== end of function ====================== */
