/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2017 Frederik Heber. All rights reserved.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * GeometryDistanceToVectorAction.cpp
 *
 *  Created on: Mar 28, 2017
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

//#include "CodePatterns/MemDebug.hpp"

#include "CodePatterns/Assert.hpp"
#include "CodePatterns/Log.hpp"

#include <string>

#include "Atom/atom.hpp"
#include "Geometry/GeometryObject.hpp"
#include "Geometry/GeometryRegistry.hpp"
#include "World.hpp"

#include "Actions/GeometryAction/GeometryDistanceToVectorAction.hpp"

using namespace MoleCuilder;

// and construct the stuff
#include "../GeometryAction/GeometryDistanceToVectorAction.def"
#include "Action_impl_pre.hpp"
/** =========== define the function ====================== */
ActionState::ptr GeometryDistanceToVectorAction::performCall() {
  // check preconditions
  World& world = World::getInstance();
  if (world.countSelectedAtoms() != 2) {
    STATUS("There must be exactly two atoms selected for GeometryAction DistanceToVector.");
    return Action::failure;
  }
  if (GeometryRegistry::getInstance().isPresentByName(params.name.get())) {
    STATUS("Name "+params.name.get()+" already present in registry.");
    return Action::failure;
  }

  World::AtomSelectionIterator iter = world.beginAtomSelection();
  const atom * const Walker = (iter++)->second;
  const atom * const OtherWalker = (iter++)->second;
  ASSERT( iter == world.endAtomSelection(),
      "GeometryDistanceToVectorAction::performCall() - selection iter not at end?");
  const Vector distance = (Walker->getPosition() - OtherWalker->getPosition())
      * (params.reverse.get() ? -1. : 1.);
  const GeometryObject vec(distance, params.name.get());
  LOG(4, "DEBUG: Adding geometry object " << vec << " with norm " << distance.Norm());
  GeometryRegistry::getInstance().addGeometry(vec);

  GeometryDistanceToVectorState *UndoState = new GeometryDistanceToVectorState(distance, params);
  return ActionState::ptr(UndoState);
}

ActionState::ptr GeometryDistanceToVectorAction::performUndo(ActionState::ptr _state) {
//  GeometryDistanceToVectorState *state = assert_cast<GeometryDistanceToVectorState*>(_state.get());

  GeometryRegistry::getInstance().removeGeometry(params.name.get());

  return ActionState::ptr(_state);
}

ActionState::ptr GeometryDistanceToVectorAction::performRedo(ActionState::ptr _state){
  GeometryDistanceToVectorState *state = assert_cast<GeometryDistanceToVectorState*>(_state.get());

  const GeometryObject vec(state->distance, params.name.get());
  LOG(4, "DEBUG: Adding geometry object " << vec);
  GeometryRegistry::getInstance().addGeometry(vec);

  return ActionState::ptr(_state);
}

bool GeometryDistanceToVectorAction::canUndo() {
  return true;
}

bool GeometryDistanceToVectorAction::shouldUndo() {
  return true;
}
/** =========== end of function ====================== */
