/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2013 Frederik Heber. All rights reserved.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * getGraph6String.cpp
 *
 *  Created on: Apr 03, 2021
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

//#include "CodePatterns/MemDebug.hpp"

#include "modules.hpp"

#include "CodePatterns/Assert.hpp"
#include "CodePatterns/Log.hpp"
#include <algorithm>
#include <string>

#include "Element/element.hpp"
#include "Graph/Graph6Writer.hpp"
#include "World.hpp"

/** =========== define the function ====================== */
std::string MoleCuilder::detail::module_getGraph6String()
{
  /*
   * Returns the current molecular graph of the selected set of atoms as
   * graph6 string representation.
   */
  Graph6Writer writer(World::getConstInstance().getSelectedAtoms());
  std::stringstream output;
  writer.write_graph6(output);
  return output.str();
}

std::string MoleCuilder::detail::module_getElementListAsString()
{
  /*
   * Returns the current molecular graph of the selected set of atoms as
   * graph6 string representation.
   */
  Graph6Writer writer(World::getConstInstance().getSelectedAtoms());
  std::stringstream output;
  writer.write_elementlist(output);
  return output.str();
}

MoleCuilder::detail::stringVec MoleCuilder::detail::module_getAllGraph6Strings()
{
  /*
   * Returns all graph6 representation by creating all permutations
   * of the non-hydrogen atoms.
   */
  std::vector<const atom *> selected_atoms = World::getConstInstance().getSelectedAtoms();
  // split set into hydrogen and non-hydrogen
  const std::vector<const atom *>::iterator partitioner =
    std::partition(selected_atoms.begin(), selected_atoms.end(), [](const atom * walker) { return walker->getElement().getAtomicNumber() != (atomicNumber_t)1; });
  std::sort(selected_atoms.begin(), partitioner);
  MoleCuilder::detail::stringVec returnlist;
  do {
    Graph6Writer writer(selected_atoms);
    std::stringstream output;
    writer.write_graph6(output);
    returnlist.push_back(output.str());
  } while(std::next_permutation(selected_atoms.begin(), partitioner));

  return returnlist;
}

MoleCuilder::detail::stringVec MoleCuilder::detail::module_getAllElementListAsStrings()
{
  /*
   * Returns non-hydrogen element lists by creating all permutations
   * of the non-hydrogen atoms.
   */
  std::vector<const atom *> selected_atoms = World::getConstInstance().getSelectedAtoms();
  // split set into non-hydrogen and hydrogen
  const std::vector<const atom *>::iterator partitioner =
    std::partition(selected_atoms.begin(), selected_atoms.end(), [](const atom * walker) { return walker->getElement().getAtomicNumber() != (atomicNumber_t)1; });
  std::sort(selected_atoms.begin(), partitioner);
  MoleCuilder::detail::stringVec returnlist;
  do {
    Graph6Writer writer(selected_atoms);
    std::stringstream output;
    writer.write_simple_elementlist(output);
    returnlist.push_back(output.str());
  } while(std::next_permutation(selected_atoms.begin(), partitioner));

  return returnlist;
}
