/*
 * listofbondsunittest.cpp
 *
 *  Created on: 18 Oct 2009
 *      Author: user
 */

using namespace std;

#include <cppunit/CompilerOutputter.h>
#include <cppunit/extensions/TestFactoryRegistry.h>
#include <cppunit/ui/text/TestRunner.h>

#include <cstring>

#include "listofbondsunittest.hpp"

#include "World.hpp"
#include "atom.hpp"
#include "bond.hpp"
#include "element.hpp"
#include "molecule.hpp"
#include "periodentafel.hpp"
#include "World.hpp"

#ifdef HAVE_TESTRUNNER
#include "UnitTestMain.hpp"
#endif /*HAVE_TESTRUNNER*/

/********************************************** Test classes **************************************/

// Registers the fixture into the 'registry'
CPPUNIT_TEST_SUITE_REGISTRATION( ListOfBondsTest );


void ListOfBondsTest::setUp()
{
  atom *Walker = NULL;

  // init private all pointers to zero
  TestMolecule = NULL;
  hydrogen = NULL;
  tafel = NULL;

  // construct element
  hydrogen = new element;
  hydrogen->Z = 1;
  strcpy(hydrogen->name, "hydrogen");
  strcpy(hydrogen->symbol, "H");


  // construct periodentafel
  tafel = World::get()->getPeriode();
  tafel->AddElement(hydrogen);

  // construct molecule (tetraeder of hydrogens)
  TestMolecule = World::get()->createMolecule();
  Walker = World::get()->createAtom();
  Walker->type = hydrogen;
  Walker->node->Init(1., 0., 1. );
  TestMolecule->AddAtom(Walker);
  Walker = World::get()->createAtom();
  Walker->type = hydrogen;
  Walker->node->Init(0., 1., 1. );
  TestMolecule->AddAtom(Walker);
  Walker = World::get()->createAtom();
  Walker->type = hydrogen;
  Walker->node->Init(1., 1., 0. );
  TestMolecule->AddAtom(Walker);
  Walker = World::get()->createAtom();
  Walker->type = hydrogen;
  Walker->node->Init(0., 0., 0. );
  TestMolecule->AddAtom(Walker);

  // check that TestMolecule was correctly constructed
  CPPUNIT_ASSERT_EQUAL( TestMolecule->AtomCount, 4 );

};


void ListOfBondsTest::tearDown()
{
  // remove
  World::get()->destroyMolecule(TestMolecule);
  // note that all the atoms, molecules, the tafel and the elements
  // are all cleaned when the world is destroyed
  World::destroy();
  MemoryUsageObserver::purgeInstance();
  logger::purgeInstance();
};

/** Unit Test of molecule::AddBond()
 *
 */
void ListOfBondsTest::AddingBondTest()
{
  bond *Binder = NULL;
  atom *atom1 = TestMolecule->start->next;
  atom *atom2 = atom1->next;
  CPPUNIT_ASSERT( atom1 != NULL );
  CPPUNIT_ASSERT( atom2 != NULL );

  // add bond
  Binder = TestMolecule->AddBond(atom1, atom2, 1);
  CPPUNIT_ASSERT( Binder != NULL );
  bond *TestBond = TestMolecule->first->next;
  CPPUNIT_ASSERT_EQUAL ( TestBond, Binder );

  // check that bond contains the two atoms
  CPPUNIT_ASSERT_EQUAL( true, Binder->Contains(atom1) );
  CPPUNIT_ASSERT_EQUAL( true, Binder->Contains(atom2) );

  // check that bond is present in both atoms
  bond *TestBond1 = *(atom1->ListOfBonds.begin());
  CPPUNIT_ASSERT_EQUAL( TestBond1, Binder );
  bond *TestBond2 = *(atom2->ListOfBonds.begin());
  CPPUNIT_ASSERT_EQUAL( TestBond2, Binder );
};

/** Unit Test of molecule::RemoveBond()
 *
 */
void ListOfBondsTest::RemovingBondTest()
{
  bond *Binder = NULL;
  atom *atom1 = TestMolecule->start->next;
  atom *atom2 = atom1->next;
  CPPUNIT_ASSERT( atom1 != NULL );
  CPPUNIT_ASSERT( atom2 != NULL );

  // add bond
  Binder = TestMolecule->AddBond(atom1, atom2, 1);
  CPPUNIT_ASSERT( Binder != NULL );

  // remove bond
  TestMolecule->RemoveBond(Binder);

  // check if removed from atoms
  CPPUNIT_ASSERT_EQUAL( (size_t) 0, atom1->ListOfBonds.size() );
  CPPUNIT_ASSERT_EQUAL( (size_t) 0, atom2->ListOfBonds.size() );

  // check if removed from molecule
  CPPUNIT_ASSERT_EQUAL( TestMolecule->first->next, TestMolecule->last );
};

/** Unit Test of molecule::RemoveBonds()
 *
 */
void ListOfBondsTest::RemovingBondsTest()
{
  bond *Binder = NULL;
  atom *atom1 = TestMolecule->start->next;
  atom *atom2 = atom1->next;
  atom *atom3 = atom2->next;
  CPPUNIT_ASSERT( atom1 != NULL );
  CPPUNIT_ASSERT( atom2 != NULL );
  CPPUNIT_ASSERT( atom3 != NULL );

  // add bond
  Binder = TestMolecule->AddBond(atom1, atom2, 1);
  CPPUNIT_ASSERT( Binder != NULL );
  Binder = TestMolecule->AddBond(atom1, atom3, 1);
  CPPUNIT_ASSERT( Binder != NULL );
  Binder = TestMolecule->AddBond(atom2, atom3, 1);
  CPPUNIT_ASSERT( Binder != NULL );

  // check that all are present
  CPPUNIT_ASSERT_EQUAL( (size_t) 2, atom1->ListOfBonds.size() );
  CPPUNIT_ASSERT_EQUAL( (size_t) 2, atom2->ListOfBonds.size() );
  CPPUNIT_ASSERT_EQUAL( (size_t) 2, atom3->ListOfBonds.size() );

  // remove bond
  TestMolecule->RemoveBonds(atom1);

  // check if removed from atoms
  CPPUNIT_ASSERT_EQUAL( (size_t) 0, atom1->ListOfBonds.size() );
  CPPUNIT_ASSERT_EQUAL( (size_t) 1, atom2->ListOfBonds.size() );
  CPPUNIT_ASSERT_EQUAL( (size_t) 1, atom3->ListOfBonds.size() );

  // check if removed from molecule
  CPPUNIT_ASSERT_EQUAL( TestMolecule->first->next, Binder );
  CPPUNIT_ASSERT_EQUAL( Binder->next, TestMolecule->last );
};

/** Unit Test of delete(bond *)
 *
 */
void ListOfBondsTest::DeleteBondTest()
{
  bond *Binder = NULL;
  atom *atom1 = TestMolecule->start->next;
  atom *atom2 = atom1->next;
  CPPUNIT_ASSERT( atom1 != NULL );
  CPPUNIT_ASSERT( atom2 != NULL );

  // add bond
  Binder = TestMolecule->AddBond(atom1, atom2, 1);
  CPPUNIT_ASSERT( Binder != NULL );

  // remove bond
  delete(Binder);

  // check if removed from atoms
  CPPUNIT_ASSERT_EQUAL( (size_t) 0, atom1->ListOfBonds.size() );
  CPPUNIT_ASSERT_EQUAL( (size_t) 0, atom2->ListOfBonds.size() );

  // check if removed from molecule
  CPPUNIT_ASSERT_EQUAL( TestMolecule->first->next, TestMolecule->last );
};

/** Unit Test of molecule::RemoveAtom()
 *
 */
void ListOfBondsTest::RemoveAtomTest()
{
  bond *Binder = NULL;
  atom *atom1 = TestMolecule->start->next;
  atom *atom2 = atom1->next;
  CPPUNIT_ASSERT( atom1 != NULL );
  CPPUNIT_ASSERT( atom2 != NULL );

  // add bond
  Binder = TestMolecule->AddBond(atom1, atom2, 1);
  CPPUNIT_ASSERT( Binder != NULL );

  // remove atom2
  TestMolecule->RemoveAtom(atom2);

  // check bond if removed from other atom
  CPPUNIT_ASSERT_EQUAL( (size_t) 0, atom1->ListOfBonds.size() );

  // check if removed from molecule
  CPPUNIT_ASSERT_EQUAL( TestMolecule->first->next, TestMolecule->last );
};

/** Unit Test of delete(atom *)
 *
 */
void ListOfBondsTest::DeleteAtomTest()
{
  bond *Binder = NULL;
  atom *atom1 = TestMolecule->start->next;
  atom *atom2 = atom1->next;
  CPPUNIT_ASSERT( atom1 != NULL );
  CPPUNIT_ASSERT( atom2 != NULL );

  // add bond
  Binder = TestMolecule->AddBond(atom1, atom2, 1);
  CPPUNIT_ASSERT( Binder != NULL );

  // remove atom2
  World::get()->destroyAtom(atom2);

  // check bond if removed from other atom
  CPPUNIT_ASSERT_EQUAL( (size_t) 0, atom1->ListOfBonds.size() );

  // check if removed from molecule
  CPPUNIT_ASSERT_EQUAL( TestMolecule->first->next, TestMolecule->last );
};
