/*
 * atomsCalculationTest.cpp
 *
 *  Created on: Feb 19, 2010
 *      Author: crueger
 */

#include "atomsCalculationTest.hpp"

#include <cppunit/CompilerOutputter.h>
#include <cppunit/extensions/TestFactoryRegistry.h>
#include <cppunit/ui/text/TestRunner.h>
#include <iostream>
#include <boost/bind.hpp>

#include "Descriptors/AtomDescriptor.hpp"
#include "Descriptors/AtomIdDescriptor.hpp"
#include "Actions/AtomsCalculation.hpp"
#include "Actions/AtomsCalculation_impl.hpp"
#include "Actions/ActionRegistry.hpp"

#include "World.hpp"
#include "World_calculations.hpp"
#include "atom.hpp"

// Registers the fixture into the 'registry'
CPPUNIT_TEST_SUITE_REGISTRATION( atomsCalculationTest );

// some stubs
class AtomStub : public atom {
public:
  AtomStub(int _id) :
  atom(),
  id(_id),
  manipulated(false)
  {}

  virtual int getId(){
    return id;
  }

  virtual void doSomething(){
    manipulated = true;
  }

  bool manipulated;
private:
  int id;
};

// set up and tear down
void atomsCalculationTest::setUp(){
  World::get();
  for(int i=0;i<ATOM_COUNT;++i){
    atoms[i]= new AtomStub(i);
    World::get()->registerAtom(atoms[i]);
  }
}
void atomsCalculationTest::tearDown(){
  World::destroy();
  ActionRegistry::purgeRegistry();
}

// some helper functions
static bool hasAll(std::vector<int> ids,int min, int max, std::set<int> excluded = std::set<int>()){
  for(int i=min;i<max;++i){
    if(!excluded.count(i)){
      std::vector<int>::iterator iter;
      bool res=false;
      for(iter=ids.begin();iter!=ids.end();++iter){
        res |= (*iter) == i;
      }
      if(!res) {
        cout << "Atom " << i << " missing in returned list" << endl;
        return false;
      }
    }
  }
  return true;
}

static bool hasNoDuplicates(std::vector<int> ids){
  std::set<int> found;
  std::vector<int>::iterator iter;
  for(iter=ids.begin();iter!=ids.end();++iter){
    int id = (*iter);
    if(found.count(id))
      return false;
    found.insert(id);
  }
  return true;
}

static void operation(atom* _atom){
  AtomStub *atom = dynamic_cast<AtomStub*>(_atom);
  assert(atom);
  atom->doSomething();
}


void atomsCalculationTest::testCalculateSimple(){
  AtomsCalculation<int> *calc = World::get()->calcOnAtoms<int>(boost::bind(&atom::getId,_1),"FOO",AllAtoms());
  std::vector<int> allIds = (*calc)();
  CPPUNIT_ASSERT(hasAll(allIds,0,ATOM_COUNT));
  CPPUNIT_ASSERT(hasNoDuplicates(allIds));
}

void atomsCalculationTest::testCalculateExcluded(){
  int excluded = ATOM_COUNT/2;
  AtomsCalculation<int> *calc = World::get()->calcOnAtoms<int>(boost::bind(&atom::getId,_1),"FOO",AllAtoms() && !AtomById(excluded));
  std::vector<int> allIds = (*calc)();
  std::set<int> excluded_set;
  excluded_set.insert(excluded);
  CPPUNIT_ASSERT(hasAll(allIds,0,ATOM_COUNT,excluded_set));
  CPPUNIT_ASSERT(hasNoDuplicates(allIds));
  CPPUNIT_ASSERT_EQUAL((size_t)(ATOM_COUNT-1),allIds.size());
}
