/*
 * analysisbondsunittest.cpp
 *
 *  Created on: Nov 7, 2009
 *      Author: heber
 */

using namespace std;

#include <cppunit/CompilerOutputter.h>
#include <cppunit/extensions/TestFactoryRegistry.h>
#include <cppunit/ui/text/TestRunner.h>

#include <iostream>
#include <stdio.h>
#include <cstring>

#include "World.hpp"
#include "analysis_bonds.hpp"
#include "analysisbondsunittest.hpp"
#include "atom.hpp"
#include "bond.hpp"
#include "bondgraph.hpp"
#include "element.hpp"
#include "molecule.hpp"
#include "periodentafel.hpp"

#ifdef HAVE_TESTRUNNER
#include "UnitTestMain.hpp"
#endif /*HAVE_TESTRUNNER*/

/********************************************** Test classes **************************************/

// Registers the fixture into the 'registry'
CPPUNIT_TEST_SUITE_REGISTRATION( AnalysisBondsTest );


void AnalysisBondsTest::setUp()
{
  atom *Walker = NULL;

  // init private all pointers to zero
  TestMolecule = NULL;
  hydrogen = NULL;
  tafel = NULL;

  // construct element
  hydrogen = new element;
  hydrogen->Z = 1;
  hydrogen->Valence = 1;
  hydrogen->NoValenceOrbitals = 1;
  strcpy(hydrogen->name, "hydrogen");
  strcpy(hydrogen->symbol, "H");
  carbon = new element;
  carbon->Z = 1;
  carbon->Valence = 4;
  carbon->NoValenceOrbitals = 4;
  strcpy(carbon->name, "carbon");
  strcpy(carbon->symbol, "C");


  // construct periodentafel
  tafel = World::get()->getPeriode();
  tafel->AddElement(hydrogen);
  tafel->AddElement(carbon);

  // construct molecule (tetraeder of hydrogens)
  TestMolecule = World::get()->createMolecule();
  Walker = World::get()->createAtom();
  Walker->type = hydrogen;
  Walker->node->Init(1.5, 0., 1.5 );
  TestMolecule->AddAtom(Walker);
  Walker = World::get()->createAtom();
  Walker->type = hydrogen;
  Walker->node->Init(0., 1.5, 1.5 );
  TestMolecule->AddAtom(Walker);
  Walker = World::get()->createAtom();
  Walker->type = hydrogen;
  Walker->node->Init(1.5, 1.5, 0. );
  TestMolecule->AddAtom(Walker);
  Walker = World::get()->createAtom();
  Walker->type = hydrogen;
  Walker->node->Init(0., 0., 0. );
  TestMolecule->AddAtom(Walker);
  Walker = World::get()->createAtom();
  Walker->type = carbon;
  Walker->node->Init(0.5, 0.5, 0.5 );
  TestMolecule->AddAtom(Walker);

  // check that TestMolecule was correctly constructed
  CPPUNIT_ASSERT_EQUAL( TestMolecule->AtomCount, 5 );

  // create a small file with table
  filename = new string("test.dat");
  ofstream test(filename->c_str());
  test << ".\tH\tC\n";
  test << "H\t1.\t1.2\n";
  test << "C\t1.2\t1.5\n";
  test.close();
  BG = new BondGraph(true);

  CPPUNIT_ASSERT_EQUAL( true , BG->LoadBondLengthTable(*filename) );
  CPPUNIT_ASSERT_EQUAL( 1., BG->GetBondLength(0,0) );
  CPPUNIT_ASSERT_EQUAL( 1.2, BG->GetBondLength(0,1) );
  CPPUNIT_ASSERT_EQUAL( 1.5, BG->GetBondLength(1,1) );

  BG->ConstructBondGraph(TestMolecule);
};


void AnalysisBondsTest::tearDown()
{
  // remove the file
  remove(filename->c_str());
  delete(filename);
  delete(BG);

  // remove molecule
  World::get()->destroyMolecule(TestMolecule);
  // note that all the atoms are cleaned by TestMolecule
  World::destroy();
};

/** UnitTest for GetMaxMinMeanBondCount().
 */
void AnalysisBondsTest::GetMaxMinMeanBondCountTest()
{
  double Min = 20.; // check that initialization resets these arbitrary values
  double Mean = 200.;
  double Max = 1e-6;
  GetMaxMinMeanBondCount(TestMolecule, Min, Mean, Max);
  CPPUNIT_ASSERT_EQUAL( 1., Min );
  CPPUNIT_ASSERT_EQUAL( 1.6, Mean );
  CPPUNIT_ASSERT_EQUAL( 4., Max );

};

/** UnitTest for MinMaxBondDistanceBetweenElements().
 */
void AnalysisBondsTest::MinMeanMaxBondDistanceBetweenElementsTest()
{
  double Min = 20.; // check that initialization resets these arbitrary values
  double Mean = 2e+6;
  double Max = 1e-6;
  double Min2 = 20.;
  double Mean2 = 2e+6;
  double Max2 = 1e-6;
  const double maxbondlength = sqrt(1.*1. + 1.*1. + .5*.5);
  const double minbondlength = sqrt(.5*.5 + .5*.5 + .5*.5);
  const double meanbondlength = (minbondlength+3.*maxbondlength)/4.;
  // check bond lengths C-H
  MinMeanMaxBondDistanceBetweenElements(TestMolecule, hydrogen, carbon, Min, Mean, Max);
  CPPUNIT_ASSERT_EQUAL( minbondlength , Min );
  CPPUNIT_ASSERT_EQUAL( meanbondlength , Mean );
  CPPUNIT_ASSERT_EQUAL( maxbondlength , Max );

  // check that elements are symmetric, i.e. C-H == H-C
  MinMeanMaxBondDistanceBetweenElements(TestMolecule, carbon, hydrogen, Min2, Mean2, Max2);
  CPPUNIT_ASSERT_EQUAL( Min , Min2 );
  CPPUNIT_ASSERT_EQUAL( Mean , Mean2 );
  CPPUNIT_ASSERT_EQUAL( Max , Max2 );

  // check no bond case (no bonds H-H in system!)
  MinMeanMaxBondDistanceBetweenElements(TestMolecule, hydrogen, hydrogen, Min, Mean, Max);
  CPPUNIT_ASSERT_EQUAL( 0. , Min );
  CPPUNIT_ASSERT_EQUAL( 0. , Mean );
  CPPUNIT_ASSERT_EQUAL( 0. , Max );
};
