/*
 * DescriptorUnittest.cpp
 *
 *  Created on: Feb 9, 2010
 *      Author: crueger
 */

#include "DescriptorUnittest.hpp"

#include <cppunit/CompilerOutputter.h>
#include <cppunit/extensions/TestFactoryRegistry.h>
#include <cppunit/ui/text/TestRunner.h>
#include <iostream>

#include <Descriptors/AtomDescriptor.hpp>
#include <Descriptors/AtomIdDescriptor.hpp>

#include "World.hpp"
#include "atom.hpp"

#ifdef HAVE_TESTRUNNER
#include "UnitTestMain.hpp"
#endif /*HAVE_TESTRUNNER*/

/********************************************** Test classes **************************************/
// Registers the fixture into the 'registry'
CPPUNIT_TEST_SUITE_REGISTRATION( DescriptorUnittest );

// set up and tear down
void DescriptorUnittest::setUp(){
  World::get();
  for(int i=0;i<ATOM_COUNT;++i){
    atoms[i]= World::get()->createAtom();
    atomIds[i] = atoms[i]->getId();
  }
}
void DescriptorUnittest::tearDown(){
  World::destroy();
}

// some helper functions
bool hasAll(std::vector<atom*> atoms,int ids[ATOM_COUNT], std::set<int> excluded = std::set<int>()){
  for(int i=0;i<ATOM_COUNT;++i){
    int id = ids[i];
    if(!excluded.count(id)){
      std::vector<atom*>::iterator iter;
      bool res=false;
      for(iter=atoms.begin();iter!=atoms.end();++iter){
        res |= (*iter)->getId() == id;
      }
      if(!res) {
        cout << "Atom " << id << " missing in returned list" << endl;
        return false;
      }
    }
  }
  return true;
}

bool hasNoDuplicates(std::vector<atom*> atoms){
  std::set<int> found;
  std::vector<atom*>::iterator iter;
  for(iter=atoms.begin();iter!=atoms.end();++iter){
    int id = (*iter)->getId();
    if(found.count(id))
      return false;
    found.insert(id);
  }
  return true;
}


void DescriptorUnittest::AtomBaseSetsTest(){
  std::vector<atom*> allAtoms = World::get()->getAllAtoms(AllAtoms());
  CPPUNIT_ASSERT_EQUAL( true , hasAll(allAtoms,atomIds));
  CPPUNIT_ASSERT_EQUAL( true , hasNoDuplicates(allAtoms));

  std::vector<atom*> noAtoms = World::get()->getAllAtoms(NoAtoms());
  CPPUNIT_ASSERT_EQUAL( true , noAtoms.empty());
}
void DescriptorUnittest::AtomIdTest(){
  // test Atoms from boundaries and middle of the set
  atom* testAtom;
  testAtom = World::get()->getAtom(AtomById(atomIds[0]));
  CPPUNIT_ASSERT(testAtom);
  CPPUNIT_ASSERT_EQUAL( atomIds[0], testAtom->getId());
  testAtom = World::get()->getAtom(AtomById(atomIds[ATOM_COUNT/2]));
  CPPUNIT_ASSERT(testAtom);
  CPPUNIT_ASSERT_EQUAL( atomIds[ATOM_COUNT/2], testAtom->getId());
  testAtom = World::get()->getAtom(AtomById(atomIds[ATOM_COUNT-1]));
  CPPUNIT_ASSERT(testAtom);
  CPPUNIT_ASSERT_EQUAL( atomIds[ATOM_COUNT-1], testAtom->getId());

  // find some ID that has not been created
  int outsideId =-1;
  bool res = false;
  while(!res) {
    ++outsideId;
    res = true;
    for(int i = 0; i < ATOM_COUNT; ++i){
      res &= atomIds[i]!=outsideId;
    }
  }
  // test from outside of set
  testAtom = World::get()->getAtom(AtomById(outsideId));
  CPPUNIT_ASSERT(!testAtom);
}
void DescriptorUnittest::AtomCalcTest(){
  // test some elementary set operations
  {
    std::vector<atom*> testAtoms = World::get()->getAllAtoms(AllAtoms()||NoAtoms());
    CPPUNIT_ASSERT_EQUAL( true , hasAll(testAtoms,atomIds));
    CPPUNIT_ASSERT_EQUAL( true , hasNoDuplicates(testAtoms));
  }

  {
    std::vector<atom*> testAtoms = World::get()->getAllAtoms(NoAtoms()||AllAtoms());
    CPPUNIT_ASSERT_EQUAL( true , hasAll(testAtoms,atomIds));
    CPPUNIT_ASSERT_EQUAL( true , hasNoDuplicates(testAtoms));
  }

  {
    std::vector<atom*> testAtoms = World::get()->getAllAtoms(NoAtoms()&&AllAtoms());
    CPPUNIT_ASSERT_EQUAL( true , testAtoms.empty());
  }

  {
    std::vector<atom*> testAtoms = World::get()->getAllAtoms(AllAtoms()&&NoAtoms());
    CPPUNIT_ASSERT_EQUAL( true , testAtoms.empty());
  }

  {
    std::vector<atom*> testAtoms = World::get()->getAllAtoms(!AllAtoms());
    CPPUNIT_ASSERT_EQUAL( true , testAtoms.empty());
  }

  {
    std::vector<atom*> testAtoms = World::get()->getAllAtoms(!NoAtoms());
    CPPUNIT_ASSERT_EQUAL( true , hasAll(testAtoms,atomIds));
    CPPUNIT_ASSERT_EQUAL( true , hasNoDuplicates(testAtoms));
  }

  // exclude and include some atoms
  {
    std::vector<atom*> testAtoms = World::get()->getAllAtoms(AllAtoms()&&(!AtomById(atomIds[ATOM_COUNT/2])));
    std::set<int> excluded;
    excluded.insert(atomIds[ATOM_COUNT/2]);
    CPPUNIT_ASSERT_EQUAL( true , hasAll(testAtoms,atomIds,excluded));
    CPPUNIT_ASSERT_EQUAL( true , hasNoDuplicates(testAtoms));
    CPPUNIT_ASSERT_EQUAL( (size_t)(ATOM_COUNT-1), testAtoms.size());
  }

  {
    std::vector<atom*> testAtoms = World::get()->getAllAtoms(NoAtoms()||(AtomById(atomIds[ATOM_COUNT/2])));
    CPPUNIT_ASSERT_EQUAL( (size_t)1, testAtoms.size());
    CPPUNIT_ASSERT_EQUAL( atomIds[ATOM_COUNT/2], testAtoms[0]->getId());
  }
}
