/*
 * TremoloParser.hpp
 *
 *  Created on: Mar 2, 2010
 *      Author: metzler
 */

#ifndef TREMOLOPARSER_HPP_
#define TREMOLOPARSER_HPP_

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <string>

#include "FormatParser.hpp"
#include "FormatParserTrait.hpp"
#include "FormatParserInterface.hpp"
#include "FormatParser_common.hpp"
#include "ParserTypes.hpp"

#include "TremoloParser_ElementKeys.hpp"
#include "TremoloKey.hpp"
#include "TremoloAtomInfoContainer.hpp"

class molecule;

// declaration of specialized FormatParserTrait
template<>
struct FormatParserTrait<tremolo>
{
  //!> Name of the parser
  static const std::string name;
  //!> suffix of the files the parser understands to read and write
  static const std::string suffix;
  //!> ParserTypes enumeration for the parser
  static const enum ParserTypes type;
};

/**
 * Loads a tremolo file into the World and saves the World as a tremolo file.
 */
template <>
class FormatParser< tremolo >  : virtual public FormatParserInterface, public FormatParser_common
{
public:
  FormatParser();
  virtual ~FormatParser();
  void load(std::istream* file);
  void save(std::ostream* file, const std::vector<atom *> &atoms);
  void setFieldsForSave(std::string atomDataLine);
  void setAtomData(const std::string &atomdata_string);

private:
  /**
   * Map to associate element keys with the respective elements.
   */
  struct ElementKeys knownTypes;

public:

  void parseKnownTypes(std::istream &file)
  {
    knownTypes.parseKnownTypes(file);
  }
  void createKnownTypesByIdentity()
  {
    knownTypes.createKnownTypesByIdentity();
  }

  /** Save given \a atoms in tremolo's exttypes file
   *
   * \note You should have stored the atoms beforehand to give them unique
   * and sensible ids.
   *
   * @param output output stream
   * @param atoms vector of atoms whose ids to store
   * @param id id to given this exttypes group
   * @return true - all atomic ids are valid, false - one id was not set (-1)
   */
  bool saveAtomsInExttypes(std::ostream &output, const std::vector<atom*> &atoms, const int id) const;

protected:
  void AtomInserted(atomId_t);
  void AtomRemoved(atomId_t);

private:
  void readAtomDataLine(std::string line);
  void readAtomDataLine(std::string line, molecule *newmol);
  void parseAtomDataKeysLine(std::string line, int offset);
  void readNeighbors(std::stringstream* line, int numberOfNeighbors, int atomId);
  void processNeighborInformation(const std::vector<atomId_t> &atoms);
  void adaptImprData();
  void adaptTorsion();
  std::string adaptIdDependentDataString(std::string data);
  bool isUsedField(std::string fieldName);
  void writeNeighbors(std::ostream* file, int numberOfNeighbors, atom* currentAtom);
  void saveLine(std::ostream* file, atom* currentAtom);

  /**
   * Map to associate the known keys with numbers.
   */
  std::map<std::string, TremoloKey::atomDataKey> knownKeys;

  /**
   * Inverse Map to have a name to each known keys.
   */
  std::map<TremoloKey::atomDataKey, std::string> knownKeyNames;

  /**
   * Fields used in the tremolo file.
   */
  std::vector<std::string> usedFields;

  /**
   * Data which is currently not stored in atoms but was provided by the input
   * file.
   */
  std::map<const atomId_t, TremoloAtomInfoContainer> additionalAtomData;

  /**
   * Default additional atom data.
   */
  TremoloAtomInfoContainer defaultAdditionalData;
};

#endif /* TREMOLOPARSER_HPP_ */
