/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2025 Frederik Heber. All rights reserved.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * QtPotentialList.cpp
 *
 *  Created on: Jun 24, 2013
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "Views/Qt4/QtPotentialList.hpp"
#include "Views/Qt4/QtNumericalItem.hpp"

#include <QtGui/QAbstractItemView>
#include <QtGui/QTreeWidget>
#include <QtGui/QTabWidget>
#include <Qt/qsettings.h>
#include <Qt/qsplitter.h>
#include <Qt/qboxlayout.h>

//#include "CodePatterns/MemDebug.hpp"

#include <iostream>

#include "CodePatterns/Log.hpp"

#include "Potentials/EmpiricalPotential.hpp"
#include "Potentials/PotentialRegistry.hpp"

using namespace std;

const int QtPotentialList::COLUMNCOUNT = COLUMNTYPES_MAX;
const char *QtPotentialList::COLUMNNAMES[QtPotentialList::COLUMNCOUNT]={"Number","Name","Charges","Parameters"};

QtPotentialList::QtPotentialList(QWidget * _parent) :
    QWidget(_parent),
    Observer("QtPotentialList"),
    potentialregistry_enabled(false)
{
  QHBoxLayout* layout = new QHBoxLayout(this);
  QSplitter *splitter = new QSplitter (Qt::Horizontal, this );
  layout->addWidget(splitter);

  // tree widget
  treewidget = new QTreeWidget (splitter);
  treewidget->setSelectionMode( QTreeWidget::SingleSelection );
  treewidget->setColumnCount(COLUMNCOUNT);
  treewidget->setSortingEnabled(true);
  //treewidget->setSizePolicy( QSizePolicy::Minimum, sizePolicy().verticalPolicy() );
  QStringList header;
  for(int i=0; i<COLUMNCOUNT;++i)
    header << COLUMNNAMES[i];
  treewidget->setHeaderLabels(header);
  treewidget->sortByColumn(0);
  splitter->addWidget(treewidget);

  dirty = true;
  
  QSettings settings;
  settings.beginGroup("QtPotentialList");
  treewidget->resize(settings.value("treewidget_size", QSize(width()/2, 20)).toSize());
  settings.endGroup();

	PotentialRegistry::getInstance().signOn(this);
	potentialregistry_enabled = true;

  connect(this,SIGNAL(changed()),this,SLOT(update()));
  connect(this,SIGNAL(needsRefill()),this,SLOT(refill()), Qt::QueuedConnection);

  emit needsRefill();
}

QtPotentialList::~QtPotentialList()
{
  QSettings settings;
  settings.beginGroup("QtPotentialList");
  settings.setValue("treewidget_size", treewidget->size());
  settings.endGroup();


  if (potentialregistry_enabled)
    PotentialRegistry::getInstance().signOff(this);
}

void QtPotentialList::update(Observable *publisher)
{
  dirty = true;

  // force an update from Qt...
//  treewidget->clear();
  emit changed(); //doesn't work!?!
}

void QtPotentialList::refill()
{
  boost::recursive_mutex::scoped_lock lock(refill_mutex);

  const PotentialRegistry &registry = PotentialRegistry::getConstInstance();

  // clear everything
  treewidget->clear();

  size_t count = 0;
  for (PotentialRegistry::const_iterator potentialiter = registry.getBeginIter();
  		potentialiter != registry.getEndIter(); ++potentialiter) {
    const EmpiricalPotential * const potential = potentialiter->second;

    // create item
    QTreeWidgetItem *treeItem = new QTreeWidgetItem(treewidget);
    treeItem->setText(NUMBER, QString::number(count));
    {
      std::stringstream output;
      output << potential->getName();
      treeItem->setText(NAME, QString(output.str().c_str()));
    }
    {
      std::stringstream output;
      output << potential->getParticleTypes();
      treeItem->setText(CHARGES, QString(output.str().c_str()));
    }
    {
      std::stringstream output;
      output << potential->getParameters();
      treeItem->setText(PARAMETERS, QString(output.str().c_str()));
    }
  }
  dirty = false;
}

void QtPotentialList::paintEvent(QPaintEvent * event)
{
  boost::recursive_mutex::scoped_lock lock(refill_mutex);

  if (dirty)
    refill();

	QWidget::paintEvent(event);
}

void QtPotentialList::subjectKilled(Observable *publisher)
{
  // as a new instance should always already be present ... just sign on
  if (static_cast<PotentialRegistry *>(publisher) == PotentialRegistry::getPointer()) {
    potentialregistry_enabled = false;
  }
}
