/*
 * Chronos.hpp
 *
 *  Created on: Mar 14, 2011
 *      Author: heber
 */

#ifndef CHRONOS_HPP_
#define CHRONOS_HPP_

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <iosfwd>
#include <map>
#include <string>

#include <boost/thread/recursive_mutex.hpp>

#include "CodePatterns/Singleton.hpp"

class ChronosTest;

class Chronos : public Singleton<Chronos>{
  //!> Singleton pattern needs private access
  friend class Singleton<Chronos>;
  //!> unit tests needs access to private members
  friend class ChronosTest;
  //!> operator needs to access some internals
  friend std::ostream& operator<<(std::ostream &ost, const Chronos &_time);
public :
  //!> typedef for the map storing times per token
  typedef std::map<const std::string, double> TimekeepingMap;

  /** Returns current kept time of function \a _name.
   *
   * @param _name name of function
   * @return current amount of time passed for this function, 0 if unknown, -1 if currently running
   */
  double getTime(const std::string &_name) const;

  /** Resets time counter for this function \a _name to zero.
   *
   * @param _name name of function
   */
  void resetTime(const std::string &_name);

  /** Starts Timing for this function \a _name.
   *
   * @param _name name of function
   */
  void startTiming(const std::string &_name);

  /** Finishes Timing for this function \a _name.
   *
   * @param _name name of function
   */
  void endTiming(const std::string &_name);

  /** Returns const reference to time keeping map.
   *
   * \return const ref to timekeeping map
   */
  const TimekeepingMap& getTimekeepingMap() const;

  /** Sums up total time accounted for.
   *
   * @return total time in seconds
   */
  double SumUpTotalTime() const;

  /** Sums up all functions accounted.
   *
   * @return number of functions
   */
  size_t SumUpTotalFunctions() const;

protected:
  /** Do not call this constructor directly, use getInstance() instead. */
  Chronos();
  /** Do not call this destructor directly, use purgeInstance() instead. */
  ~Chronos();

private:
  /** Returns the current timer in seconds.
   *
   * This function is present to allow for changing of time measurements
   * without the other functions noticing.
   *
   * \note We convert clock() by CLOCKS_PER_SEC.
   * \note We convert struct tms::tms_utime by _SC_CLK_TCK via sysconf().
   *
   * @return current time in unit of seconds
   */
  double getCurrentTime() const;

  typedef std::pair<size_t, size_t> sec_ncsec_t;

  /** Returns the correct difference between split tuples (sec, nsec).
   *
   * \param _time1 first tuple
   * \param _time2 second tuple
   */
  static
  double calculateCorrectTimeDifference(
  		const sec_ncsec_t &_time1,
      const sec_ncsec_t &_time2);

  //!> typedef for the map storing status of time keeping per token
  typedef std::map<const std::string, double> TimerMap;

  //!> typedef for the map storing number of recursive calls to this token
  typedef std::map<const std::string, size_t> TimerRecursionMap;

  //!> map storing times per token
  TimekeepingMap AccountedTime;
  //!> map storing time keeping status per token
  TimerMap StartingTime;
  //!> map storing level of recursion per token
  TimerRecursionMap RecursionMap;

#ifdef HAVE_TIME_H
  timespec basetime;
#else
#ifdef HAVE_SYS_TIME_H
  struct timezone basetime;
#else
#ifdef HAVE_SYS_TIMES_H
  struct tms *basetime;
#endif
#endif
#endif

  //!> mutex for keeping chronos timekeeping atomic
  mutable boost::recursive_mutex ChronosMutex;
};

/** Output for Chronos.
 *
 * @param ost output stream
 * @param _time Chronos to print
 * @return reference to given output stream for concatenation
 */
std::ostream& operator<<(std::ostream &ost, const Chronos &_time);

// inline functions

inline
const Chronos::TimekeepingMap& Chronos::getTimekeepingMap() const
{
  return AccountedTime;
}

#endif /* CHRONOS_HPP_ */
