/*
 * Berendsen.cpp
 *
 *  Created on: Aug 20, 2010
 *      Author: crueger
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "CodePatterns/MemDebug.hpp"

#include "Berendsen.hpp"
#include "Element/element.hpp"
#include "config.hpp"
#include "CodePatterns/Verbose.hpp"
#include "CodePatterns/Log.hpp"
#include "Helpers/defs.hpp"
#include "ThermoStatContainer.hpp"
#include "World.hpp"

Berendsen::Berendsen(double _TempFrequency) :
  TempFrequency(_TempFrequency)
{}

Berendsen::Berendsen() :
  TempFrequency(2.5)
{}

Berendsen::~Berendsen()
{}

const char *ThermostatTraits<Berendsen>::name = "Berendsen";

std::string ThermostatTraits<Berendsen>::getName(){
  return ThermostatTraits<Berendsen>::name;
}

Thermostat *ThermostatTraits<Berendsen>::make(class ConfigFileBuffer * const fb){
  double TempFrequency;
  const int verbose = 0;
  ParseForParameter(verbose,fb,"Thermostat", 0, 2, 1, double_type, &TempFrequency, 1, critical); // read \tau_T
  return new Berendsen(TempFrequency);
}

double Berendsen::scaleAtoms(unsigned int step,double ActualTemp,ATOMSET(std::list) atoms){
  return doScaleAtoms(step,ActualTemp,atoms.begin(),atoms.end());
}

double Berendsen::scaleAtoms(unsigned int step,double ActualTemp,ATOMSET(std::vector) atoms){
  return doScaleAtoms(step,ActualTemp,atoms.begin(),atoms.end());
}

double Berendsen::scaleAtoms(unsigned int step,double ActualTemp,ATOMSET(std::set) atoms){
  return doScaleAtoms(step,ActualTemp,atoms.begin(),atoms.end());
}

template <class ForwardIterator>
double Berendsen::doScaleAtoms(unsigned int step,double ActualTemp,ForwardIterator begin, ForwardIterator end){
  LOG(2,  "Applying Berendsen-VanGunsteren thermostat...");
  double ekin=0;
  double ScaleTempFactor = getContainer().TargetTemp/ActualTemp;
  for(ForwardIterator iter=begin;iter!=end;++iter){
    Vector U = (*iter)->getAtomicVelocityAtStep(step);
    if ((*iter)->getFixedIon() == 0) { // even FixedIon moves, only not by other's forces
      U *= sqrt(1+(World::getInstance().getConfig()->Deltat/TempFrequency)*(ScaleTempFactor-1));
      ekin += 0.5*(*iter)->getType()->getMass() * U.NormSquared();
    }
    (*iter)->setAtomicVelocityAtStep(step, U);
  }
  return ekin;
}

std::string Berendsen::name(){
  return ThermostatTraits<Berendsen>::name;
}

std::string Berendsen::writeParams(){
  stringstream sstr;
  sstr << TempFrequency;
  return sstr.str();
}
