/*
 * NoseHoover.cpp
 *
 *  Created on: Aug 20, 2010
 *      Author: crueger
 */

#include "NoseHoover.hpp"

#include "element.hpp"
#include "config.hpp"
#include "Helpers/Verbose.hpp"
#include "Helpers/Log.hpp"
#include "ThermoStatContainer.hpp"
#include "World.hpp"

NoseHoover::NoseHoover(double _HooverMass) :
  HooverMass(_HooverMass),
  alpha(0)
{}

NoseHoover::NoseHoover() :
  HooverMass(0.)
{}

NoseHoover::~NoseHoover()
{}

ThermostatTraits<class NoseHoover>::ThermostatTraits() :
  name("NoseHoover")
{}

Thermostat *ThermostatTraits<class NoseHoover>::make(class ConfigFileBuffer * const fb){
  double HooverMass;
  const int verbose = 0;
  ParseForParameter(verbose,fb,"Thermostat", 0, 2, 1, double_type, &HooverMass, 1, critical); // read Hoovermass
  return new class NoseHoover(HooverMass);
}

double NoseHoover::scaleAtoms(unsigned int step,double ActualTemp,ATOMSET(std::list) atoms){
  return doScaleAtoms(step,ActualTemp,atoms.begin(),atoms.end());
}

double NoseHoover::scaleAtoms(unsigned int step,double ActualTemp,ATOMSET(std::vector) atoms){
  return doScaleAtoms(step,ActualTemp,atoms.begin(),atoms.end());
}

double NoseHoover::scaleAtoms(unsigned int step,double ActualTemp,ATOMSET(std::set) atoms){
  return doScaleAtoms(step,ActualTemp,atoms.begin(),atoms.end());
}

template <class ForwardIterator>
double NoseHoover::doScaleAtoms(unsigned int step,double ActualTemp,ForwardIterator begin, ForwardIterator end){
  DoLog(2) && (Log() << Verbose(2) <<  "Applying Nose-Hoover thermostat..." << endl);
  init(step,begin,end);
  delta_alpha = (delta_alpha - (3.*count+1.) * getContainer().TargetTemp)/(HooverMass*Units2Electronmass);
  alpha += delta_alpha*World::getInstance().getConfig()->Deltat;
  DoLog(3) && (Log() << Verbose(3) << "alpha = " << delta_alpha << " * " << World::getInstance().getConfig()->Deltat << " = " << alpha << "." << endl);
  double ekin =0;
  for(ForwardIterator iter=begin;iter!=end;++iter){
    Vector &U = (*iter)->Trajectory.U.at(step);
    if ((*iter)->FixedIon == 0) { // even FixedIon moves, only not by other's forces
      U += World::getInstance().getConfig()->Deltat/(*iter)->getType()->mass * (alpha * (U * (*iter)->getType()->mass));
      ekin += (0.5*(*iter)->getType()->mass) * U.NormSquared();
    }
  }
  return ekin;
}

template <class ForwardIterator>
void NoseHoover::init(unsigned int step,ForwardIterator begin, ForwardIterator end){
  delta_alpha=0;
  count=0;
  for(ForwardIterator iter = begin;iter!=end;++iter){
    Vector &U = (*iter)->Trajectory.U.at(step);
    if ((*iter)->FixedIon == 0) { // even FixedIon moves, only not by other's forces
      delta_alpha += U.NormSquared()*(*iter)->getType()->mass;
    }
    ++count;
  }
}

std::string NoseHoover::name(){
  return ThermostatTraits<NoseHoover>().name;
}

std::string NoseHoover::writeParams(){
  std::stringstream sstr;
  sstr << HooverMass;
  return sstr.str();
}
