/*
 * PdbParser.hpp
 *
 *  Created on: Aug 17, 2010
 *      Author: heber
 */

#ifndef PDBPARSER_HPP_
#define PDBPARSER_HPP_

#include <string>
#include "Parser/FormatParser.hpp"

/**
 * Known keys for the ATOMDATA line.
 */
class TremoloKey {
public:
  enum atomDataKey {
    noKey,
    x,
    u,
    F,
    stress,
    Id,
    neighbors,
    imprData,
    GroupMeasureTypeNo,
    Type,
    extType,
    name,
    resName,
    chainID,
    resSeq,
    occupancy,
    tempFactor,
    segID,
    Charge,
    charge,
    GrpTypeNo,
    torsion
  };
};

/**
 * Holds tremolo-specific information which is not store in the atom class.
 */
class TremoloAtomInfoContainer {
public:
  TremoloAtomInfoContainer();
  void set(TremoloKey::atomDataKey key, std::string value);
  std::string get(TremoloKey::atomDataKey key);
  std::string F;
  std::string stress;
  std::string imprData;
  std::string GroupMeasureTypeNo;
  std::string extType;
  std::string name;
  std::string resName;
  std::string chainID;
  std::string resSeq;
  std::string occupancy;
  std::string tempFactor;
  std::string segID;
  std::string Charge;
  std::string charge;
  std::string GrpTypeNo;
  std::string torsion;
  std::vector<int> neighbors;
};

/**
 * Loads a tremolo file into the World and saves the World as a tremolo file.
 */
class PdbParser : public FormatParser
{
public:
  PdbParser();
  ~PdbParser();
  void load(std::istream* file);
  void save(std::ostream* file);
  void setFieldsForSave(std::string atomDataLine);


private:
  void readAtomDataLine(string line);
  void parseAtomDataKeysLine(string line, int offset);
  void readNeighbors(std::stringstream* line, int numberOfNeighbors, int atomId);
  void processNeighborInformation();
  void adaptImprData();
  void adaptTorsion();
  std::string adaptIdDependentDataString(std::string data);
  bool isUsedField(std::string fieldName);
  void writeNeighbors(std::ostream* file, int numberOfNeighbors, atom* currentAtom);
  void saveLine(std::ostream* file, atom* currentAtom);

  /**
   * Map to associate the known keys with numbers.
   */
  std::map<std::string, TremoloKey::atomDataKey> knownKeys;

  /**
   * Fields used in the tremolo file.
   */
  std::vector<std::string> usedFields;

  /**
   * Data which is currently not stored in atoms but was provided by the input
   * file.
   */
  std::map<int, TremoloAtomInfoContainer> additionalAtomData;

  /**
   * Default additional atom data.
   */
  TremoloAtomInfoContainer defaultAdditionalData;

  /**
   * Maps original atom IDs received from the parsed file to atom IDs in the
   * world.
   */
  std::map<int, int> atomIdMap;
};

#endif /* PDBPARSER_HPP_ */
