/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * CommandLineParser_validate.cpp
 *
 *  Created on: Nov 8, 2010
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "Helpers/MemDebug.hpp"

#include <iostream>
#include <string>

#include "Actions/Values.hpp"
#include "CommandLineParser_validate.hpp"

/** boost::program_options validator specialization for VectorValue.
 * \param &v  reference for return value
 * \param &values string vector of scanned options
 * \param *
 * \param
 *
 */
void validate(boost::any& v, const std::vector<std::string>& values, VectorValue *, int)
{
  VectorValue VV;
  std::vector<std::string> components;

  // split comma-separated values
  if (values.size() != 1) {
    std::cerr <<  "Not one vector but " << values.size() << " given " << std::endl;
    throw boost::program_options::validation_error("Unequal to one vector given");
  }
  std::string argument(values.at(0));
  std::string::iterator Aiter = argument.begin();
  std::string::iterator Biter = argument.begin();
  for (; Aiter != argument.end(); ++Aiter) {
    if (*Aiter == ',') {
      components.push_back(std::string(Biter,Aiter));
      do {
        Aiter++;
      } while (*Aiter == ' ' || *Aiter == '\t');
      Biter = Aiter;
    }
  }
  components.push_back(std::string(Biter,argument.end()));

  if (components.size() != 3) {
    std::cerr <<  "Specified vector does not have three components but " << components.size() << std::endl;
    throw boost::program_options::validation_error("Specified vector does not have three components");
  }
  VV.x = boost::lexical_cast<double>(components.at(0));
  VV.y = boost::lexical_cast<double>(components.at(1));
  VV.z = boost::lexical_cast<double>(components.at(2));
  v = boost::any(VectorValue(VV));
}

void validate(boost::any& v, const std::vector<std::string>& values, BoxValue *, int)
{
  BoxValue BV;
  std::vector<std::string> components;

  // split comma-separated values
  if (values.size() != 1) {
    std::cerr <<  "Not one vector but " << values.size() << " given " << std::endl;
    throw boost::program_options::validation_error("Unequal to one vector given");
  }
  std::string argument(values.at(0));
  std::string::iterator Aiter = argument.begin();
  std::string::iterator Biter = argument.begin();
  for (; Aiter != argument.end(); ++Aiter) {
    if (*Aiter == ',') {
      components.push_back(std::string(Biter,Aiter));
      do {
        Aiter++;
      } while (*Aiter == ' ' || *Aiter == '\t');
      Biter = Aiter;
    }
  }
  components.push_back(std::string(Biter,argument.end()));

  if (components.size() != 6) {
    std::cerr <<  "Specified vector does not have three components but " << components.size() << std::endl;
    throw boost::program_options::validation_error("Specified symmetric box matrix does not have six components");
  }
  BV.xx = boost::lexical_cast<double>(components.at(0));
  BV.yx = boost::lexical_cast<double>(components.at(1));
  BV.yy = boost::lexical_cast<double>(components.at(2));
  BV.zx = boost::lexical_cast<double>(components.at(3));
  BV.zy = boost::lexical_cast<double>(components.at(4));
  BV.zz = boost::lexical_cast<double>(components.at(5));
  v = boost::any(BoxValue(BV));
}

/** boost::program_options validator specialization for boost::filesystem::path.
 * \param &v  reference for return value
 * \param &values string vector of scanned options
 * \param *
 * \param
 *
 */
void validate(boost::any& v, const std::vector<std::string>& values, boost::filesystem::path *, int)
{
  boost::filesystem::path filename;

  std::cerr << "boost::filesystem::path validator used." << std::endl;

  // split comma-separated values
  if (values.size() != 1) {
    std::cerr <<  "Not one file but " << values.size() << " given " << std::endl;
    throw boost::program_options::validation_error("Unequal to one file given");
  }
  filename = values.at(0);
  v = boost::any(boost::filesystem::path(filename));
}

