/*
 * World.hpp
 *
 *  Created on: Feb 3, 2010
 *      Author: crueger
 */

#ifndef WORLD_HPP_
#define WORLD_HPP_

/*********************************************** includes ***********************************/

#include <string>
#include <map>
#include <vector>
#include <set>
#include <boost/thread.hpp>
#include <boost/shared_ptr.hpp>

#include "types.hpp"
#include "Descriptors/SelectiveIterator.hpp"
#include "Patterns/Observer.hpp"
#include "Patterns/Cacheable.hpp"
#include "Patterns/Singleton.hpp"

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

// forward declarations
class config;
class periodentafel;
class MoleculeListClass;
class atom;
class molecule;
class AtomDescriptor;
class AtomDescriptor_impl;
class MoleculeDescriptor;
class MoleculeDescriptor_impl;
class ManipulateAtomsProcess;
template<typename T>
class AtomsCalculation;

/****************************************** forward declarations *****************************/

/********************************************** Class World *******************************/

class World : public Singleton<World>, public Observable
{

// Make access to constructor and destructor possible from inside the singleton
friend class Singleton<World>;

// necessary for coupling with descriptors
friend class AtomDescriptor_impl;
friend class AtomDescriptor;
friend class MoleculeDescriptor_impl;
friend class MoleculeDescriptor;

// Actions, calculations etc associated with the World
friend class ManipulateAtomsProcess;
template<typename> friend class AtomsCalculation;
public:

  // Types for Atom and Molecule structures
  typedef std::map<atomId_t,atom*> AtomSet;
  typedef std::map<moleculeId_t,molecule*> MoleculeSet;

  /***** getter and setter *****/
  // reference to pointer is used for legacy reason... reference will be removed latter to keep encapsulation of World object
  /**
   * returns the periodentafel for the world.
   */
  periodentafel *&getPeriode();

  /**
   * returns the configuration for the world.
   */
  config *&getConfig();

  /**
   * returns the first atom that matches a given descriptor.
   * Do not rely on ordering for descriptors that match more than one atom.
   */
  atom* getAtom(AtomDescriptor descriptor);

  /**
   * returns a vector containing all atoms that match a given descriptor
   */
  std::vector<atom*> getAllAtoms(AtomDescriptor descriptor);
  std::vector<atom*> getAllAtoms();

  /**
   * returns a calculation that calls a given function on all atoms matching a descriptor.
   * the calculation is not called at this point and can be used as an action, i.e. be stored in
   * menus, be kept around for later use etc.
   */
  template<typename T> AtomsCalculation<T>* calcOnAtoms(boost::function<T(atom*)>,std::string,AtomDescriptor);
  template<typename T> AtomsCalculation<T>* calcOnAtoms(boost::function<T(atom*)>,std::string);

  /**
   * get the number of atoms in the World
   */
  int numAtoms();

  /**
   * returns the first molecule that matches a given descriptor.
   * Do not rely on ordering for descriptors that match more than one molecule.
   */
  molecule *getMolecule(MoleculeDescriptor descriptor);

  /**
   * returns a vector containing all molecules that match a given descriptor
   */
  std::vector<molecule*> getAllMolecules(MoleculeDescriptor descriptor);
  std::vector<molecule*> getAllMolecules();

  /**
   * get the number of molecules in the World
   */
  int numMolecules();

  /**
   * get the domain size as a symmetric matrix (6 components)
   */
  double * getDomain();

  /**
   * set the domain size as a symmetric matrix (6 components)
   */
  void setDomain(double * matrix);

  /**
   * get the default name
   */
  std::string getDefaultName();

  /**
   * set the default name
   */
  void setDefaultName(std::string name);

  /*
   * get the ExitFlag
   */
  int getExitFlag();

  /*
   * set the ExitFlag
   */
  void setExitFlag(int flag);

  /***** Methods to work with the World *****/

  /**
   * create a new molecule. This method should be used whenever any kind of molecule is needed. Assigns a unique
   * ID to the molecule and stores it in the World for later retrieval. Do not create molecules directly.
   */
  molecule *createMolecule();

  void destroyMolecule(molecule*);
  void destroyMolecule(moleculeId_t);

  /**
   * Create a new atom. This method should be used whenever any atom is needed. Assigns a unique ID and stores
   * the atom in the World. If the atom is not destroyed it will automatically be destroyed when the world ends.
   */
  atom *createAtom();

  /**
   * Registers a Atom unknown to world. Needed in some rare cases, e.g. when cloning atoms, or in some unittests.
   * Do not re-register Atoms already known to the world since this will cause double-frees.
   */
  int registerAtom(atom*);

  /**
     * Delete some atom and erase it from the world. Use this whenever you need to destroy any atom. Do not call delete on
     * atom directly since this will leave the pointer inside the world.
   */
  void destroyAtom(atom*);

  /**
   * Delete some atom and erase it from the world. Use this whenever you need to destroy any atom. Do not call delete on
   * atom directly since this will leave the pointer inside the world.
   */
  void destroyAtom(atomId_t);

  /**
   * used when changing an atom Id.
   * Unless you are calling this method from inside an atom don't fiddle with the third parameter.
   *
   * Return value indicates wether the change could be done or not.
   */
  bool changeAtomId(atomId_t oldId, atomId_t newId, atom* target=0);

  /**
   * Produces a process that calls a function on all Atoms matching a given descriptor. The process is not
   * called at this time, so it can be passed around, stored inside menuItems etc.
   */
  ManipulateAtomsProcess* manipulateAtoms(boost::function<void(atom*)>,std::string,AtomDescriptor);
  ManipulateAtomsProcess* manipulateAtoms(boost::function<void(atom*)>,std::string);

protected:
  /**** Iterators to use internal data structures */

  // Atoms
  typedef SelectiveIterator<atom*,AtomSet,AtomDescriptor> AtomIterator;

  /**
   * returns an iterator over all Atoms matching a given descriptor.
   * used for internal purposes, like AtomProcesses and AtomCalculations.
   */
  AtomIterator getAtomIter(AtomDescriptor descr);

  /**
   * returns an iterator to the end of the AtomSet. Due to overloading this iterator
   * can be compared to iterators produced by getAtomIter (see the mis-matching types).
   * Thus it can be used to detect when such an iterator is at the end of the list.
   * used for internal purposes, like AtomProcesses and AtomCalculations.
   */
  AtomIterator atomEnd();

  // Molecules

  typedef SelectiveIterator<molecule*,MoleculeSet,MoleculeDescriptor> MoleculeIterator;

  /**
   * returns an iterator over all Molecules matching a given descriptor.
   * used for internal purposes, like MoleculeProcesses and MoleculeCalculations.
   */
  MoleculeIterator getMoleculeIter(MoleculeDescriptor descr);

  /**
   * returns an iterator to the end of the MoleculeSet. Due to overloading this iterator
   * can be compared to iterators produced by getMoleculeIter (see the mis-matching types).
   * Thus it can be used to detect when such an iterator is at the end of the list.
   * used for internal purposes, like MoleculeProcesses and MoleculeCalculations.
   */
  MoleculeIterator moleculeEnd();


  /******* Internal manipulation routines for double callback and Observer mechanism ******/
  void doManipulate(ManipulateAtomsProcess *);

private:

  atomId_t getNextAtomId();
  void releaseAtomId(atomId_t);
  bool reserveAtomId(atomId_t);

  periodentafel *periode;
  config *configuration;
  static double *cell_size;
  std::string defaultName;
  int ExitFlag;
public:
  AtomSet atoms;
private:
  std::set<atomId_t> atomIdPool; //<!stores the pool for all available AtomIds below currAtomId
  atomId_t currAtomId; //!< stores the next available Id for atoms
  MoleculeSet molecules;
  moleculeId_t currMoleculeId;
private:
  /**
   * private constructor to ensure creation of the world using
   * the singleton pattern.
   */
  World();

  /**
   * private destructor to ensure destruction of the world using the
   * singleton pattern.
   */
  virtual ~World();

  /*****
   * some legacy stuff that is include for now but will be removed later
   *****/
public:
  MoleculeListClass *&getMolecules();

private:
  MoleculeListClass *molecules_deprecated;
};

#endif /* WORLD_HPP_ */
