/*
 * QSeisData.cpp
 *
 *  Created on: Jan 30, 2011
 *      Author: landvogt
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "QSeisData.hpp"

#include "UIElements/Views/Qt4/Plotting/QSeisPlotCurve.hpp"
#include "UIElements/Views/Qt4/Plotting/QSeisCurveRegistry.hpp"

// have this after(!) all Qt includes
#include "CodePatterns/MemDebug.hpp"

/** Constructor for QSeisData.
 *
 *  We create the QSeisPlotCurve instance here by calling createCurve().
 *
 * @param name file name data is associated with
 * @param columnNo column number in the file this instance represents
 */
QSeisData::QSeisData(QString name, quint32 columnNo) :
         intName(name), columnNumber(columnNo)
{
    createCurve();
}

/** Constructor for QSeisData.
 *
 * Note that we do not spawn any QSeisCurve instance here.
 *
 * @param xValues predefined set of x values
 * @param yValues predefined set of y values
 * @param name file for this data set
 */
QSeisData::QSeisData(std::vector<double> xValues, std::vector<double> yValues, QString name) :
        intName(name), xData(xValues), yData(yValues)
{

}

/** Destructor for QSeisData.
 *
 */
QSeisData::~QSeisData()
{
}

/** Create QSeisCurve instance for this data set if not yet present.
 *
 * We check in QSeisCurveRegistry whether a curve associated with \a intName is
 * already present. If so, we associated with it, overwriting any association
 * which another QSeisData has (probably from an old file that was deleted but
 * whose data has been preserved so far). If not, we create a new instance and
 * register it.
 *
 * \note We make this instance child of the curve such that it is destroyed when
 *       then curve is destroyed.
 *
 */
void QSeisData::createCurve()
{
    QSeisPlotCurve *new_curve = NULL;
    // if curve is already present, replace binding to this one
    if (QSeisCurveRegistry::getInstance().isPresentByName(intName.toStdString())) {
        new_curve = QSeisCurveRegistry::getInstance().getByName(intName.toStdString());
    } else {
//	QString type = QSeisXMLParser::getInstance().getTypeOfName(intName);
//        new_curve = new QSeisPlotCurve(intName, type);
    }
    connect(this, SIGNAL(dataChanged(QwtData*)), new_curve, SLOT(updateCurve(QwtData*)));
    connect(this, SIGNAL(dataOutOfDate()), new_curve, SLOT(markAsOutOfDate()));
    connect(new_curve, SIGNAL(dataRequested()), this, SIGNAL(dataRequested()));
    setParent(new_curve);

    // if replacing present curve, notify
    if (QSeisCurveRegistry::getInstance().isPresentByName(intName.toStdString())) {
        emit dataChanged(this);
    } else {
        QSeisCurveRegistry::getInstance().registerInstance(new_curve);
    }
}

/** Slot for updating of the data contents.
 *
 *  We get here all data because then QSeisParser (who gives the signal) can
 *  bind to all QSeisData instances' slots at once.
 *
 * @param newData new data to store
 */
void QSeisData::updateData(const std::vector<std::vector<double> > &newData)
{
    xData.clear();
    yData.clear();
    if (newData.size() > columnNumber)
    {
        xData = newData[0];
        yData = newData[columnNumber];
    }
    emit dataChanged((QwtData *)this);
}

/** Copy function.
 *
 * This is defined in QwtData, hence we need to implement it.
 *
 * @return copy of this instance
 */
QSeisData* QSeisData::copy() const
{
    return new QSeisData(xData, yData, intName);
}

/** Getter for size of the data set, i.e. number of tupels.
 *
 * @return number of data tupels.
 */
size_t QSeisData::size() const
{
    return xData.size();
}

/** Getter for specific x value
 *
 * @param i index of value
 * @return \a xData[i] or 0. if not present
 */
double QSeisData::x(size_t i) const
{
    if (i < xData.size())
        return xData.at(i);
    else
        return 0.0f;
}

/** Getter for specific y value
 *
 * @param i index of value
 * @return \a yData[i] or 0. if not present
 */
double QSeisData::y(size_t i) const
{
    if (i < yData.size())
        return yData.at(i);
    else
        return 0.0f;
}
