/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * LinkedCellUnitTest.cpp
 *
 *  Created on: Apr 9, 2010
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

using namespace std;

#include <cppunit/CompilerOutputter.h>
#include <cppunit/extensions/TestFactoryRegistry.h>
#include <cppunit/ui/text/TestRunner.h>

#include <iostream>
#include <stdio.h>
#include <cstring>

#include "atom.hpp"
#include "Descriptors/MoleculeDescriptor.hpp"
#include "element.hpp"
#include "linkedcell.hpp"
#include "molecule.hpp"
#include "periodentafel.hpp"
#include "PointCloudAdaptor.hpp"
#include "World.hpp"

#include "LinkedCellUnitTest.hpp"

#ifdef HAVE_TESTRUNNER
#include "UnitTestMain.hpp"
#endif /*HAVE_TESTRUNNER*/

/********************************************** Test classes **************************************/

// Registers the fixture into the 'registry'
CPPUNIT_TEST_SUITE_REGISTRATION( LinkedCellTest );


void LinkedCellTest::setUp()
{
  atom *Walker = NULL;

  // construct element
  hydrogen = World::getInstance().getPeriode()->FindElement(1);
  CPPUNIT_ASSERT(hydrogen != NULL && "could not find element hydrogen");

  // construct molecule (water molecule)
  TestMolecule = World::getInstance().createMolecule();
  CPPUNIT_ASSERT(TestMolecule != NULL && "could not create molecule");
  for (double x=0.5;x<3;x+=1.)
    for (double y=0.5;y<3;y+=1.)
      for (double z=0.5;z<3;z+=1.) {
        Walker = World::getInstance().createAtom();
        CPPUNIT_ASSERT(Walker != NULL && "could not create atom");
        Walker->setType(hydrogen);
        Walker->setPosition(Vector(x, y, z ));
        TestMolecule->AddAtom(Walker);
      }

  // construct linked cell
  PointCloudAdaptor<molecule> cloud(TestMolecule);
  LC = new LinkedCell (cloud, 1.);
  CPPUNIT_ASSERT(LC != NULL && "could not create LinkedCell");

  // check that TestMolecule was correctly constructed
  CPPUNIT_ASSERT_EQUAL( TestMolecule->getAtomCount(), 3*3*3 );
};


void LinkedCellTest::tearDown()
{
  delete(LC);
  World::purgeInstance();
};


/** UnitTest for LinkedCell::CheckBounds().
 */
void LinkedCellTest::CheckBoundsTest()
{
  // check for within bounds
  LC->n[0] = LC->n[1] = LC->n[2] = 0;
  CPPUNIT_ASSERT_EQUAL( true, LC->CheckBounds() );
  LC->n[0] = LC->n[1] = LC->n[2] = 1;
  CPPUNIT_ASSERT_EQUAL( true, LC->CheckBounds() );
  LC->n[0] = LC->n[1] = LC->n[2] = 2;
  CPPUNIT_ASSERT_EQUAL( true, LC->CheckBounds() );

  // check for out of bounds
  cout << "The following test is supposed to fail and produce an ERROR." << endl;
  LC->n[0] = 404040;
  CPPUNIT_ASSERT_EQUAL( false, LC->CheckBounds() );
  cout << "The following test is supposed to fail and produce an ERROR." << endl;
  LC->n[0] = 0;
  LC->n[1] = 5000;
  CPPUNIT_ASSERT_EQUAL( false, LC->CheckBounds() );
  cout << "The following test is supposed to fail and produce an ERROR." << endl;
  LC->n[1] = 0;
  LC->n[2] = -70;
  CPPUNIT_ASSERT_EQUAL( false, LC->CheckBounds() );
  cout << "The following test is supposed to fail and produce an ERROR." << endl;
  LC->n[0] = LC->n[1] = LC->n[2] = 3;
  CPPUNIT_ASSERT_EQUAL( false, LC->CheckBounds() );
};


/** UnitTest for LinkedCell::GetCurrentCell().
 * Note that CheckBounds() is used and has to be tested already.
 */
void LinkedCellTest::GetCurrentCellTest()
{
  // within bounds
  LC->n[0] = LC->n[1] = LC->n[2] = 0;
  CPPUNIT_ASSERT_EQUAL( (const TesselPointSTLList*)&LC->LC[0 * 3*3 + 0 * 3 + 0], LC->GetCurrentCell() );
  LC->n[0] = LC->n[1] = LC->n[2] = 1;
  CPPUNIT_ASSERT_EQUAL( (const TesselPointSTLList*)&LC->LC[1 * 3*3 + 1 * 3 + 1], LC->GetCurrentCell() );
  LC->n[0] = LC->n[1] = LC->n[2] = 2;
  CPPUNIT_ASSERT_EQUAL( (const TesselPointSTLList*)&LC->LC[2 * 3*3 + 2 * 3 + 2], LC->GetCurrentCell() );

  // out of bounds
  LC->n[0] = LC->n[1] = LC->n[2] = 3;
  cout << "The following test is supposed to fail and produce an ERROR." << endl;
  CPPUNIT_ASSERT_EQUAL( (const TesselPointSTLList*)NULL, LC->GetCurrentCell() );
  LC->n[0] = LC->n[1] = LC->n[2] = -1;
  cout << "The following test is supposed to fail and produce an ERROR." << endl;
  CPPUNIT_ASSERT_EQUAL( (const TesselPointSTLList*)NULL, LC->GetCurrentCell() );
};

/** UnitTest for LinkedCell::GetRelativeToCurrentCell().
 */
void LinkedCellTest::GetRelativeToCurrentCellTest()
{
  int offset[3];

  // offset to (0,0,0) always
  offset[0] = offset[1] = offset[2] = 0;
  LC->n[0] = LC->n[1] = LC->n[2] = 0;
  CPPUNIT_ASSERT_EQUAL( (const TesselPointSTLList*)&LC->LC[0], LC->GetRelativeToCurrentCell(offset) );
  offset[0] = offset[1] = offset[2] = -1;
  LC->n[0] = LC->n[1] = LC->n[2] = 1;
  CPPUNIT_ASSERT_EQUAL( (const TesselPointSTLList*)&LC->LC[0], LC->GetRelativeToCurrentCell(offset) );
  offset[0] = offset[1] = offset[2] = -2;
  LC->n[0] = LC->n[1] = LC->n[2] = 2;
  CPPUNIT_ASSERT_EQUAL( (const TesselPointSTLList*)&LC->LC[0], LC->GetRelativeToCurrentCell(offset) );

  // offset to (0,0,0) - 1.*(x/y/z) out of bounds
  offset[0] = offset[1] = offset[2] = 0;
  offset[0] = -1;
  LC->n[0] = LC->n[1] = LC->n[2] = 0;
  cout << "The following test is supposed to fail and produce an ERROR." << endl;
  CPPUNIT_ASSERT_EQUAL( (const TesselPointSTLList*)NULL, LC->GetRelativeToCurrentCell(offset) );
  offset[0] = offset[1] = offset[2] = 0;
  offset[1] = -1;
  LC->n[0] = LC->n[1] = LC->n[2] = 0;
  cout << "The following test is supposed to fail and produce an ERROR." << endl;
  CPPUNIT_ASSERT_EQUAL( (const TesselPointSTLList*)NULL, LC->GetRelativeToCurrentCell(offset) );
  offset[0] = offset[1] = offset[2] = 0;
  offset[2] = -1;
  LC->n[0] = LC->n[1] = LC->n[2] = 0;
  cout << "The following test is supposed to fail and produce an ERROR." << endl;
  CPPUNIT_ASSERT_EQUAL( (const TesselPointSTLList*)NULL, LC->GetRelativeToCurrentCell(offset) );

  // out of bounds
  offset[0] = offset[1] = offset[2] = -5054932;
  LC->n[0] = LC->n[1] = LC->n[2] = 1;
  cout << "The following test is supposed to fail and produce an ERROR." << endl;
  CPPUNIT_ASSERT_EQUAL( (const TesselPointSTLList*)NULL, LC->GetRelativeToCurrentCell(offset) );
  offset[0] = offset[1] = offset[2] = 192345;
  LC->n[0] = LC->n[1] = LC->n[2] = 1;
  cout << "The following test is supposed to fail and produce an ERROR." << endl;
  CPPUNIT_ASSERT_EQUAL( (const TesselPointSTLList*)NULL, LC->GetRelativeToCurrentCell(offset) );

  // index is out of bounds, offset points within
  offset[0] = offset[1] = offset[2] = -2;
  LC->n[0] = LC->n[1] = LC->n[2] = 4;
  CPPUNIT_ASSERT_EQUAL( (const TesselPointSTLList*)&LC->LC[2 * 3*3 + 2 * 3 + 2], LC->GetRelativeToCurrentCell(offset) );

  // index is within bounds, offset points out
  offset[0] = offset[1] = offset[2] = 2;
  LC->n[0] = LC->n[1] = LC->n[2] = 2;
  cout << "The following test is supposed to fail and produce an ERROR." << endl;
  CPPUNIT_ASSERT_EQUAL( (const TesselPointSTLList*)NULL, LC->GetRelativeToCurrentCell(offset) );
};


/** UnitTest for LinkedCell::SetIndexToNode().
 */
void LinkedCellTest::SetIndexToNodeTest()
{
  // check all atoms
  for(molecule::iterator iter = TestMolecule->begin(); iter != TestMolecule->end();++iter){
    CPPUNIT_ASSERT_EQUAL( true, LC->SetIndexToNode(*iter) );
  }

  // check internal vectors, returns false, because this atom is not in LC-list!
  atom *newAtom = World::getInstance().createAtom();
  newAtom->setName("test");
  newAtom->setPosition(Vector(1,1,1));
  CPPUNIT_ASSERT_EQUAL( false, LC->SetIndexToNode(newAtom) );
  World::getInstance().destroyAtom(newAtom);

  // check out of bounds vectors
  newAtom = World::getInstance().createAtom();
  newAtom->setName("test");
  newAtom->setPosition(Vector(0,-1,0));
  CPPUNIT_ASSERT_EQUAL( false, LC->SetIndexToNode(newAtom) );
  World::getInstance().destroyAtom(newAtom);
};


/** UnitTest for LinkedCell::SetIndexToVector().
 */
void LinkedCellTest::SetIndexToVectorTest()
{
  Vector tester;

  // check center of each cell
  for (double x=0.5;x<3;x+=1.)
    for (double y=0.5;y<3;y+=1.)
      for (double z=0.5;z<3;z+=1.) {
        tester = Vector(x,y,z);
        CPPUNIT_ASSERT_EQUAL( true, LC->SetIndexToVector(tester) );
      }
  // check corners of each cell
  for (double x=1.;x<4;x+=1.)
    for (double y=1.;y<4;y+=1.)
      for (double z=1.;z<4;z+=1.) {
        tester= Vector(x,y,z);
        cout << "Tester is at " << tester << "." << endl;
        CPPUNIT_ASSERT_EQUAL( true, LC->SetIndexToVector(tester) );
      }
  // check out of bounds
  for (double x=0.5-1e-10;x<5;x+=3.1)
    for (double y=0.5-1e-10;y<5;y+=3.1)
      for (double z=0.5-1e-10;z<5;z+=3.1) {
        tester = Vector(x,y,z);
        cout << "The following test is supposed to fail and produce an ERROR." << endl;
        CPPUNIT_ASSERT_EQUAL( false, LC->SetIndexToVector(tester) );
      }
  // check nonsense vectors
  tester= Vector(-423598,3245978,29349);
  cout << "The following test is supposed to fail and produce an ERROR." << endl;
  CPPUNIT_ASSERT_EQUAL( false, LC->SetIndexToVector(tester) );
};


/** UnitTest for LinkedCell::GetNeighbourBounds().
 */
void LinkedCellTest::GetNeighbourBoundsTest()
{
  Vector tester;
  int lower[NDIM], upper[NDIM];

  tester= Vector(0.5,0.5,0.5);
  LC->SetIndexToVector(tester);
  LC->GetNeighbourBounds(lower, upper);
  for (int i=0;i<NDIM;i++)
    CPPUNIT_ASSERT_EQUAL( 0, lower[i]);
  for (int i=0;i<NDIM;i++)
    CPPUNIT_ASSERT_EQUAL( 1, upper[i]);
};


/** UnitTest for LinkedCell::GetallNeighbours().
 */
void LinkedCellTest::GetallNeighboursTest()
{
  Vector tester;
  TesselPointSTLList *ListOfPoints = NULL;
  size_t size = 0;

  // get all atoms
  tester= Vector(1.5,1.5,1.5);
  CPPUNIT_ASSERT_EQUAL ( true, LC->SetIndexToVector(tester) );
  ListOfPoints = LC->GetallNeighbours();
  size = ListOfPoints->size();
  CPPUNIT_ASSERT_EQUAL( (size_t)27, size );

  for(molecule::iterator iter = TestMolecule->begin(); iter != TestMolecule->end(); ++iter){
    ListOfPoints->remove((*iter));
    size--;
    CPPUNIT_ASSERT_EQUAL( size, ListOfPoints->size() );
  }
  CPPUNIT_ASSERT_EQUAL( (size_t)0, size );
  CPPUNIT_ASSERT_EQUAL( (size_t)0, ListOfPoints->size() );
  CPPUNIT_ASSERT_EQUAL( true, ListOfPoints->empty() );
  delete(ListOfPoints);

  // get all atoms in one corner
  tester= Vector(0.5, 0.5, 0.5);
  CPPUNIT_ASSERT_EQUAL ( true, LC->SetIndexToVector(tester) );
  ListOfPoints = LC->GetallNeighbours();
  size=ListOfPoints->size();
  CPPUNIT_ASSERT_EQUAL( (size_t)8, size );
  for(molecule::iterator iter = TestMolecule->begin(); iter != TestMolecule->end(); ++iter){
    if (((*iter)->at(0) <2) && ((*iter)->at(1) <2) && ((*iter)->at(2) <2)) {
      ListOfPoints->remove(*iter);
      size--;
      CPPUNIT_ASSERT_EQUAL( size, ListOfPoints->size() );
    }
  }
  CPPUNIT_ASSERT_EQUAL( (size_t)0, size );
  CPPUNIT_ASSERT_EQUAL( (size_t)0, ListOfPoints->size() );
  CPPUNIT_ASSERT_EQUAL( true, ListOfPoints->empty() );
  delete(ListOfPoints);

  // get all atoms from one corner
  tester = Vector(0.5, 0.5, 0.5);
  CPPUNIT_ASSERT_EQUAL ( true, LC->SetIndexToVector(tester) );
  ListOfPoints = LC->GetallNeighbours(3);
  size=ListOfPoints->size();
  CPPUNIT_ASSERT_EQUAL( (size_t)27, size );
  for(molecule::iterator iter = TestMolecule->begin(); iter!=TestMolecule->end();++iter){
    ListOfPoints->remove(*iter);
    size--;
    CPPUNIT_ASSERT_EQUAL( size, ListOfPoints->size() );
  }
  CPPUNIT_ASSERT_EQUAL( (size_t)0, size );
  CPPUNIT_ASSERT_EQUAL( (size_t)0, ListOfPoints->size() );
  CPPUNIT_ASSERT_EQUAL( true, ListOfPoints->empty() );
  delete(ListOfPoints);
};


/** UnitTest for LinkedCell::GetPointsInsideSphere().
 */
void LinkedCellTest::GetPointsInsideSphereTest()
{
  Vector tester;
  TesselPointSTLList *ListOfPoints = NULL;
  size_t size = 0;

  // get all points around central arom with radius 1.
  tester= Vector(1.5,1.5,1.5);
  CPPUNIT_ASSERT_EQUAL ( true, LC->SetIndexToVector(tester) );
  ListOfPoints = LC->GetPointsInsideSphere(1., &tester);
  size = ListOfPoints->size();
  CPPUNIT_ASSERT_EQUAL( (size_t)7, size );
  for(molecule::iterator iter = TestMolecule->begin(); iter!=TestMolecule->end();++iter){
    if (((*iter)->DistanceSquared(tester) - 1.) < MYEPSILON ) {
      ListOfPoints->remove(*iter);
      size--;
      CPPUNIT_ASSERT_EQUAL( size, ListOfPoints->size() );
    }
  }
  CPPUNIT_ASSERT_EQUAL( (size_t)0, size );
  CPPUNIT_ASSERT_EQUAL( (size_t)0, ListOfPoints->size() );
  CPPUNIT_ASSERT_EQUAL( true, ListOfPoints->empty() );
  delete(ListOfPoints);
};
