/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * AnalysisCorrelationToSurfaceUnitTest.cpp
 *
 *  Created on: Oct 13, 2009
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

using namespace std;

#include <cppunit/CompilerOutputter.h>
#include <cppunit/extensions/TestFactoryRegistry.h>
#include <cppunit/ui/text/TestRunner.h>

#include <cstring>

#include "analysis_correlation.hpp"
#include "atom.hpp"
#include "boundary.hpp"
#include "CodePatterns/Assert.hpp"
#include "Descriptors/MoleculeDescriptor.hpp"
#include "element.hpp"
#include "molecule.hpp"
#include "linkedcell.hpp"
#include "periodentafel.hpp"
#include "PointCloudAdaptor.hpp"
#include "tesselation.hpp"
#include "World.hpp"

#include "AnalysisCorrelationToSurfaceUnitTest.hpp"

#ifdef HAVE_TESTRUNNER
#include "UnitTestMain.hpp"
#endif /*HAVE_TESTRUNNER*/

/********************************************** Test classes **************************************/

// Registers the fixture into the 'registry'
CPPUNIT_TEST_SUITE_REGISTRATION( AnalysisCorrelationToSurfaceUnitTest );

void AnalysisCorrelationToSurfaceUnitTest::setUp()
{
  ASSERT_DO(Assert::Throw);

  setVerbosity(5);

  atom *Walker = NULL;

  // init private all pointers to zero
  TestSurfaceMolecule = NULL;
  surfacemap = NULL;
  binmap = NULL;
  Surface = NULL;
  LC = NULL;

  // prepare element list
  hydrogen = World::getInstance().getPeriode()->FindElement(1);
  CPPUNIT_ASSERT(hydrogen != NULL && "hydrogen element not found");
  elements.clear();

  // construct molecule (tetraeder of hydrogens) base
  TestSurfaceMolecule = World::getInstance().createMolecule();

  Walker = World::getInstance().createAtom();
  Walker->setType(hydrogen);
  Walker->setPosition(Vector(1., 0., 1. ));
  TestSurfaceMolecule->AddAtom(Walker);

  Walker = World::getInstance().createAtom();
  Walker->setType(hydrogen);
  Walker->setPosition(Vector(0., 1., 1. ));
  TestSurfaceMolecule->AddAtom(Walker);

  Walker = World::getInstance().createAtom();
  Walker->setType(hydrogen);
  Walker->setPosition(Vector(1., 1., 0. ));
  TestSurfaceMolecule->AddAtom(Walker);

  Walker = World::getInstance().createAtom();
  Walker->setType(hydrogen);
  Walker->setPosition(Vector(0., 0., 0. ));
  TestSurfaceMolecule->AddAtom(Walker);

  // check that TestMolecule was correctly constructed
  CPPUNIT_ASSERT_EQUAL( TestSurfaceMolecule->getAtomCount(), 4 );

  TestSurfaceMolecule->ActiveFlag = true;

  // init tesselation and linked cell
  Surface = new Tesselation;
  PointCloudAdaptor<molecule> cloud(TestSurfaceMolecule);
  LC = new LinkedCell(cloud, 5.);
  FindNonConvexBorder(TestSurfaceMolecule, Surface, (const LinkedCell *&)LC, 2.5, NULL);

  // add outer atoms
  carbon = World::getInstance().getPeriode()->FindElement(6);
  TestSurfaceMolecule = World::getInstance().createMolecule();
  Walker = World::getInstance().createAtom();
  Walker->setType(carbon);
  Walker->setPosition(Vector(4., 0., 4. ));
  TestSurfaceMolecule->AddAtom(Walker);

  Walker = World::getInstance().createAtom();
  Walker->setType(carbon);
  Walker->setPosition(Vector(0., 4., 4. ));
  TestSurfaceMolecule->AddAtom(Walker);

  Walker = World::getInstance().createAtom();
  Walker->setType(carbon);
  Walker->setPosition(Vector(4., 4., 0. ));
  TestSurfaceMolecule->AddAtom(Walker);

  // add inner atoms
  Walker = World::getInstance().createAtom();
  Walker->setType(carbon);
  Walker->setPosition(Vector(0.5, 0.5, 0.5 ));
  TestSurfaceMolecule->AddAtom(Walker);

  World::getInstance().selectAllMolecules(AllMolecules());
  allMolecules = World::getInstance().getSelectedMolecules();
  CPPUNIT_ASSERT_EQUAL( (size_t) 2, allMolecules.size());

  // init maps
  surfacemap = NULL;
  binmap = NULL;

};


void AnalysisCorrelationToSurfaceUnitTest::tearDown()
{
  if (surfacemap != NULL)
    delete(surfacemap);
  if (binmap != NULL)
    delete(binmap);

  delete(Surface);
  // note that all the atoms are cleaned by TestMolecule
  delete(LC);
  World::purgeInstance();
  logger::purgeInstance();
};


/** Checks whether setup() does the right thing.
 */
void AnalysisCorrelationToSurfaceUnitTest::SurfaceTest()
{
  CPPUNIT_ASSERT_EQUAL( 4, TestSurfaceMolecule->getAtomCount() );
  CPPUNIT_ASSERT_EQUAL( (size_t)2, allMolecules.size() );
  CPPUNIT_ASSERT_EQUAL( (size_t)4, Surface->PointsOnBoundary.size() );
  CPPUNIT_ASSERT_EQUAL( (size_t)6, Surface->LinesOnBoundary.size() );
  CPPUNIT_ASSERT_EQUAL( (size_t)4, Surface->TrianglesOnBoundary.size() );
};

void AnalysisCorrelationToSurfaceUnitTest::CorrelationToSurfaceTest()
{
  // do the pair correlation
  elements.push_back(hydrogen);
  surfacemap = CorrelationToSurface( allMolecules, elements, Surface, LC );
//  OutputCorrelationToSurface ( (ofstream *)&cout, surfacemap );
  CPPUNIT_ASSERT( surfacemap != NULL );
  CPPUNIT_ASSERT_EQUAL( (size_t)4, surfacemap->size() );
};

void AnalysisCorrelationToSurfaceUnitTest::CorrelationToSurfaceHydrogenBinNoRangeTest()
{
  BinPairMap::iterator tester;
  elements.push_back(hydrogen);
  surfacemap = CorrelationToSurface( allMolecules, elements, Surface, LC );
  // put pair correlation into bins and check with no range
//  OutputCorrelationToSurface ( (ofstream *)&cout, surfacemap );
  binmap = BinData( surfacemap, 0.5, 0., 0. );
  CPPUNIT_ASSERT_EQUAL( (size_t)1, binmap->size() );
  OutputCorrelationMap<BinPairMap> ( (ofstream *)&cout, binmap, OutputCorrelation_Header, OutputCorrelation_Value );
  tester = binmap->begin();
  CPPUNIT_ASSERT_EQUAL( 0., tester->first );
  CPPUNIT_ASSERT_EQUAL( 4, tester->second );

};

void AnalysisCorrelationToSurfaceUnitTest::CorrelationToSurfaceHydrogenBinRangeTest()
{
  BinPairMap::iterator tester;
  elements.push_back(hydrogen);
  surfacemap = CorrelationToSurface( allMolecules, elements, Surface, LC );
//  OutputCorrelationToSurface ( (ofstream *)&cout, surfacemap );
  // ... and check with [0., 2.] range
  binmap = BinData( surfacemap, 0.5, 0., 2. );
  CPPUNIT_ASSERT_EQUAL( (size_t)5, binmap->size() );
//  OutputCorrelation ( (ofstream *)&cout, binmap );
  tester = binmap->begin();
  CPPUNIT_ASSERT_EQUAL( 0., tester->first );
  CPPUNIT_ASSERT_EQUAL( 4, tester->second );
  tester = binmap->find(1.);
  CPPUNIT_ASSERT_EQUAL( 1., tester->first );
  CPPUNIT_ASSERT_EQUAL( 0, tester->second );

};

void AnalysisCorrelationToSurfaceUnitTest::CorrelationToSurfaceCarbonBinNoRangeTest()
{
  BinPairMap::iterator tester;
  elements.push_back(carbon);
  surfacemap = CorrelationToSurface( allMolecules, elements, Surface, LC );
//  OutputCorrelationToSurface ( (ofstream *)&cout, surfacemap );
  // put pair correlation into bins and check with no range
  binmap = BinData( surfacemap, 0.5, 0., 0. );
  //OutputCorrelation ( (ofstream *)&cout, binmap );
  CPPUNIT_ASSERT_EQUAL( (size_t)9, binmap->size() );
  // inside point is first and must have negative value
  tester = binmap->lower_bound(4.25-0.5); // start depends on the min value and
  CPPUNIT_ASSERT( tester != binmap->end() );
  CPPUNIT_ASSERT_EQUAL( 3, tester->second );
  // inner point
  tester = binmap->lower_bound(0.);
  CPPUNIT_ASSERT( tester != binmap->end() );
  CPPUNIT_ASSERT_EQUAL( 1, tester->second );
};

void AnalysisCorrelationToSurfaceUnitTest::CorrelationToSurfaceCarbonBinRangeTest()
{
  BinPairMap::iterator tester;
  elements.push_back(carbon);
  surfacemap = CorrelationToSurface( allMolecules, elements, Surface, LC );
//  OutputCorrelationToSurface ( (ofstream *)&cout, surfacemap );
  // ... and check with [0., 2.] range
  binmap = BinData( surfacemap, 0.5, -2., 4. );
  //OutputCorrelation ( (ofstream *)&cout, binmap );
  CPPUNIT_ASSERT_EQUAL( (size_t)13, binmap->size() );
  // three outside points
  tester = binmap->lower_bound(4.25-0.5);
  CPPUNIT_ASSERT( tester != binmap->end() );
  CPPUNIT_ASSERT_EQUAL( 3, tester->second );
  // inner point
  tester = binmap->lower_bound(0.);
  CPPUNIT_ASSERT( tester != binmap->end() );
  CPPUNIT_ASSERT_EQUAL( 1, tester->second );
};
