/*
 * Registry_impl.hpp
 *
 *  Created on: Jul 28, 2010
 *      Author: heber
 */

#ifndef REGISTRY_IMPL_HPP_
#define REGISTRY_IMPL_HPP_

#include "Helpers/MemDebug.hpp"

#include "Patterns/Registry.hpp"
#include "Patterns/Singleton_impl.hpp"

#include "Helpers/Assert.hpp"
#include <iosfwd>

/** Constructor for class Registry.
 */
template <class T> Registry<T>::Registry()
{}

/** Destructor for class Registry.
 */
template <class T> Registry<T>::~Registry()
{
  typename std::map<const std::string,T*>::iterator iter;
  for(iter=InstanceMap.begin();iter!=InstanceMap.end();++iter) {
    delete iter->second;
  }
  InstanceMap.clear();
}

/** Returns pointer to an instance named by \a name.
 * \param name name of instance
 * \return pointer to instance
 */
template <class T> T* Registry<T>::getByName(const std::string name){
  typename std::map<const std::string,T*>::iterator iter;
  iter = InstanceMap.find(name);
  ASSERT(iter!=InstanceMap.end(),"Query for an instance not stored in registry");
  return iter->second;
}

/** States whether instance is present or not.
 * \note This is needed as Registry<T>::getByName() ASSERT()s that instance is in std::map.
 * \param name name of instance
 * \return true - v present, false - instance absent
 */
template <class T>bool Registry<T>::isPresentByName(const std::string name){
  typename std::map<const std::string,T*>::iterator iter;
  iter = InstanceMap.find(name);
  return iter!=InstanceMap.end();
}

/** Registers an instance with the Registry.
 * \param *instance pointer to T.
 */
template <class T>void Registry<T>::registerInstance(T* instance){
  std::pair<typename std::map<const std::string,T*>::iterator,bool> ret;
  //cout << "Trying to register instance with name " << instance->getName() << "." << endl;
  ret = InstanceMap.insert(typename std::pair<const std::string,T*>(instance->getName(),instance));
  ASSERT(ret.second,"Two instances with the same name added to registry");
}

/** Unregisters an instance.
 * \param *instance pointer to instance.
 */
template <class T>void Registry<T>::unregisterInstance(T* instance){
  //cout << "Unregistering instance with name " << instance->getName() << "." << endl;
  InstanceMap.erase(instance->getName());
}

/** Returns an iterator pointing to the start of the std::map of instance's.
 * \return begin iterator
 */
template <class T>
typename std::map<const std::string,T*>::iterator Registry<T>::getBeginIter()
{
  return InstanceMap.begin();
}

/** Returns an iterator pointing to the end of the std::map of instance's.
 * \return end iterator
 */
template <class T>
typename std::map<const std::string,T*>::iterator Registry<T>::getEndIter()
{
  return InstanceMap.end();
}

/** Returns a const iterator pointing to the start of the std::map of instance's.
 * \return constant begin iterator
 */
template <class T>
typename std::map<const std::string,T*>::const_iterator Registry<T>::getBeginIter() const
{
  return InstanceMap.begin();
}

/** Returns a const iterator pointing to the end of the std::map of instance's.
 * \return constant end iterator
 */
template <class T>
typename std::map<const std::string,T*>::const_iterator Registry<T>::getEndIter() const
{
  return InstanceMap.end();
}

/** Prints the contents of the Registry \a &m to \a &ost.
 * \param &ost output stream
 * \param &m reference to Registry
 * \return reference to the above out stream for concatenation
 */
template <class T>
std::ostream& operator<<(std::ostream& ost, const Registry<T>& m)
{
  ost << "Registry contains:" << std::endl;
  for (typename std::map<const std::string,T*>::const_iterator iter = m.getBeginIter(); iter != m.getEndIter(); ++iter) {
    ost << "\t" << iter->first << " with pointer " << iter->second << std::endl;
  }
  return ost;
};

/**
 * This define allows simple instantiation of the necessary registryfunctions
 * at a chosen place.
 */
#define CONSTRUCT_REGISTRY(name) \
    template name* Registry<name>::getByName(const std::string name); \
    template bool Registry<name>::isPresentByName(const std::string name); \
    template void Registry<name>::registerInstance(name*); \
    template void Registry<name>::unregisterInstance(name*); \
    template std::map<const std::string,name*>::iterator Registry<name>::getBeginIter(); \
    template std::map<const std::string,name*>::const_iterator Registry<name>::getBeginIter() const; \
    template std::map<const std::string,name*>::iterator Registry<name>::getEndIter(); \
    template std::map<const std::string,name*>::const_iterator Registry<name>::getEndIter() const;


#endif /* REGISTRY_IMPL_HPP_ */
