/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010-2012 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * QtElementList.cpp
 *
 *  Created on: Mar 6, 2012
 *      Author: ankele
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "Views/Qt4/QtElementList.hpp"

#include <iostream>

#include "CodePatterns/MemDebug.hpp"

#include "Atom/atom.hpp"
#include "Formula.hpp"
#include "molecule.hpp"
#include "MoleculeListClass.hpp"

#include "Element/element.hpp"
#include "Element/periodentafel.hpp"
#include "Descriptors/AtomTypeDescriptor.hpp"
#include <QAbstractItemView>
#include "Actions/SelectionAction/Atoms/AtomByElementAction.hpp"
#include "Actions/SelectionAction/Atoms/NotAtomByElementAction.hpp"

using namespace std;

const int QtElementList::COLUMNCOUNT = COLUMNTYPES_MAX;
const char *QtElementList::COLUMNNAMES[QtElementList::COLUMNCOUNT]={"Number","Name","Symbol","Mass","Occurrence"};

QtElementList::QtElementList(QWidget * _parent) :
    QTreeWidget (_parent),
    Observer("QtElementList")
{
  setColumnCount(COLUMNCOUNT);
  setSelectionMode(QAbstractItemView::MultiSelection);

  QStringList header;
	for(int i=0; i<COLUMNCOUNT;++i)
	  header << COLUMNNAMES[i];
	setHeaderLabels(header);

	dirty = true;
	refill();

	molecules = World::getInstance().getMolecules();
	molecules->signOn(this);

  connect(this,SIGNAL(itemSelectionChanged()),this,SLOT(rowSelected()));
  connect(this,SIGNAL(changed()),this,SLOT(update()));
}

QtElementList::~QtElementList()
{
	molecules->signOff(this);
}

void QtElementList::update(Observable *publisher) {

  dirty = true;

  // force an update from Qt...
  clear();
  //emit changed(); doesn't work!?!
}

void QtElementList::refill() {

  periodentafel *&periode = World::getInstance().getPeriode();

  elementSelection.clear();

  int i;

  clear();
  periodentafel::const_iterator iter;
  for(iter = periode->begin(),i=0;
      iter != periode->end();
      ++i,++iter) {
    const element *e = iter->second;
    int count = 0;
    count = World::getInstance().getAllAtoms(AtomByType(e)).size();

    QTreeWidgetItem *treeItem = new QTreeWidgetItem(this);
    treeItem->setText(NUMBER, QString::number(e->getAtomicNumber()));
    treeItem->setText(NAME, QString(e->getName().c_str()));
    treeItem->setText(SYMBOL, QString(e->getSymbol().c_str()));
    treeItem->setText(MASS, QString::number(e->getMass()));
    if (count > 0){
      treeItem->setText(OCCURRENCE, QString::number(count));
    }else{
      treeItem->setText(OCCURRENCE, "none");
      treeItem->setDisabled(true);
    }
    elementSelection.push_back(treeItem->isSelected());
  }
  dirty = false;
}

void QtElementList::paintEvent(QPaintEvent * event)
{
  if (dirty)
    refill();
  QTreeWidget::paintEvent(event);
}

void QtElementList::subjectKilled(Observable *publisher) {
}


void QtElementList::rowSelected()
{
  //std::cout << "rowSelected\n";
  periodentafel *&periode = World::getInstance().getPeriode();

  for (int i=0;i<topLevelItemCount();i++){
    QTreeWidgetItem *item = topLevelItem(i);
    bool newSelection = item->isSelected();
    if (newSelection != elementSelection[i]){
      elementSelection[i] = newSelection;

      int atomicNumber = item->text(NUMBER).toInt();
      const element *e = periode->FindElement(atomicNumber);
      if (newSelection)
        MoleCuilder::SelectionAtomByElement(e);
      else
        MoleCuilder::SelectionNotAtomByElement(e);
    }
  }
}

