/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * CheckAgainstAdjacencyFile.cpp
 *
 *  Created on: Mar 3, 2011
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "CodePatterns/MemDebug.hpp"

#include <iostream>

#include "CheckAgainstAdjacencyFile.hpp"

#include "atom.hpp"
#include "Bond/bond.hpp"
#include "CodePatterns/Assert.hpp"
#include "CodePatterns/Log.hpp"
#include "CodePatterns/Verbose.hpp"
#include "Helpers/defs.hpp"

CheckAgainstAdjacencyFile::CheckAgainstAdjacencyFile() :
  CurrentBonds(new int[MAXBONDS]),
  status(true),
  NonMatchNumber(0)
{
  for(int i=0;i<MAXBONDS;i++)
    CurrentBonds[i] = 0;
}

CheckAgainstAdjacencyFile::~CheckAgainstAdjacencyFile()
{
  delete[](CurrentBonds);
}

void CheckAgainstAdjacencyFile::CompareBonds(const atom *&Walker, size_t &CurrentBondsOfAtom, int AtomNr, std::map<int, atom*> &ListOfAtoms)
{
  size_t j = 0;
  int id = -1;

  //Log() << Verbose(2) << "Walker is " << *Walker << ", bond partners: ";
  const BondList& ListOfBonds = Walker->getListOfBonds();
  if (CurrentBondsOfAtom == ListOfBonds.size()) {
    for (BondList::const_iterator Runner = ListOfBonds.begin();
        Runner != ListOfBonds.end();
        ++Runner) {
      id = (*Runner)->GetOtherAtom(Walker)->getNr();
      j = 0;
      for (; (j < CurrentBondsOfAtom) && (CurrentBonds[j++] != id);)
        ; // check against all parsed bonds
      if (CurrentBonds[j - 1] != id) { // no match ? Then mark in ListOfAtoms
        ListOfAtoms[AtomNr] = NULL;
        NonMatchNumber++;
        status = false;
        ELOG(2, id << " can not be found in list." << endl);
      } else {
        //Log() << Verbose(0) << "[" << id << "]\t";
      }
    }
    //Log() << Verbose(0) << endl;
  } else {
    LOG(0, "STATUS: Number of bonds for Atom " << *Walker << " does not match, parsed " << CurrentBondsOfAtom << " against " << ListOfBonds.size() << ".");
    status = false;
  }
}
;

/** Checks contents of adjacency file against bond structure in structure molecule.
 * \param File file to parser
 * \param ListOfAtoms map from int (index in file) to atom
 * \return true - structure is equal, false - not equivalence
 */
bool CheckAgainstAdjacencyFile::operator()(std::ifstream &File, std::map<int, atom*> ListOfAtoms)
{
  LOG(0, "STATUS: Looking at bond structure stored in adjacency file and comparing to present one ... ");
  if (File.fail()) {
    LOG(1, "STATUS: Adjacency file not found." << endl);
    return false;
  }

  char buffer[MAXSTRINGSIZE];
  int tmp;
  // Parse the file line by line and count the bonds
  while (!File.eof()) {
    File.getline(buffer, MAXSTRINGSIZE);
    stringstream line;
    line.str(buffer);
    int AtomNr = -1;
    line >> AtomNr;
    size_t CurrentBondsOfAtom = -1; // we count one too far due to line end
    // parse into structure
    if (AtomNr >= 0) {
      ASSERT(ListOfAtoms.count(AtomNr),
          "CheckAgainstAdjacencyFile::operator() - index "
          +toString(AtomNr)+" not present in ListOfAtoms.");
      const atom *Walker = ListOfAtoms[AtomNr];
      while (line >> ws >> tmp) {
        LOG(3, "INFO: Recognized bond partner " << tmp);
        CurrentBonds[++CurrentBondsOfAtom] = tmp;
        ASSERT(CurrentBondsOfAtom < MAXBONDS,
            "molecule::CheckAdjacencyFileAgainstMolecule() - encountered more bonds than allowed: "
            +toString(CurrentBondsOfAtom)+" >= "+toString(int(MAXBONDS))+"!");
      }
      // compare against present bonds
      CompareBonds(Walker, CurrentBondsOfAtom, AtomNr, ListOfAtoms);
    } else {
      if (AtomNr != -1)
        ELOG(2, AtomNr << " is negative.");
    }
  }

  if (status) { // if equal we parse the KeySetFile
    LOG(0, "STATUS: Equal.");
  } else
    LOG(0, "STATUS: Not equal by " << NonMatchNumber << " atoms.");
  return status;
}
;
