/*
 * Formula.hpp
 *
 *  Created on: Jul 21, 2010
 *      Author: crueger
 */

#ifndef FORMULA_HPP_
#define FORMULA_HPP_

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif


#include <vector>
#include <iosfwd>
#include <iterator>

#include "Exceptions/FormulaStringParseException.hpp"

#include "types.hpp"

#include "CodePatterns/enumeration.hpp"

class element;


class Formula
{
public:
  typedef const element*                                key_type;
  typedef unsigned int                                  mapped_type;
  typedef std::pair<key_type, mapped_type>              value_type;
  typedef std::vector<unsigned int>                     set_t;

  template <class result_type>
  class _iterator : public std::iterator <std::bidirectional_iterator_tag,
                                         result_type>
  {
  public:
    _iterator(set_t&);
    _iterator(set_t&,size_t);
    _iterator(const _iterator &rhs);
    ~_iterator();
    _iterator   &operator=(const _iterator &);
    bool         operator==(const _iterator&);
    bool         operator!=(const _iterator&);
    _iterator    operator++();
    _iterator    operator++(int);
    _iterator    operator--();
    _iterator    operator--(int);
    result_type  operator*();
    result_type *operator->();
  private:
    set_t *set;
    size_t pos; // cannot use an iterator, because we need a position to access the element
  };

  typedef _iterator<value_type>                 iterator;
  typedef _iterator<const value_type>           const_iterator;

  Formula();
  Formula(const std::string&);
  Formula(const Formula&);
  virtual ~Formula();

  Formula &operator=(const Formula&);

  std::string toString() const;
  void fromString(const std::string&) throw(FormulaStringParseException);
  bool checkOut(std::ostream*) const;

  unsigned int getElementCount() const;
  bool hasElement(key_type)             const;
  bool hasElement(atomicNumber_t)       const;
  bool hasElement(const std::string&)   const;

  void operator+=(key_type);
  void operator+=(atomicNumber_t);
  void operator+=(const std::string&);

  void operator-=(key_type);
  void operator-=(atomicNumber_t);
  void operator-=(const std::string&);

  void addElements(key_type,unsigned int);
  void addElements(atomicNumber_t,unsigned int);
  void addElements(const std::string&,unsigned int);

  void addFormula(const Formula&,unsigned int);

  enumeration<key_type> enumerateElements() const;

  // only const versions, because someone might try to increment a previously
  // not set element
  const unsigned int operator[](key_type) const;
  const unsigned int operator[](atomicNumber_t) const;
  const unsigned int operator[](std::string)    const;

  bool operator==(const Formula&) const;
  bool operator!=(const Formula&) const;

  iterator       begin();
  const_iterator begin() const;
  iterator       end();
  const_iterator end()   const;

  void clear();

private:
  void parseFromString(std::string::const_iterator&,std::string::const_iterator&,char) throw(FormulaStringParseException);
  int parseMaybeNumber(std::string::const_iterator &it,std::string::const_iterator &end) throw(FormulaStringParseException);
  // this contains all counts of elements in the formula
  // the size of the actual structure might be used in comparisons
  // so be sure not to keep any zero elements at the end
  // e.g. {5;4;0;0;0;2;1} or {0;1;0;0;2} are ok, but {1;2;3;0;0;0} is bad
  set_t elementCounts;
  unsigned int numElements;
};

std::ostream &operator<<(std::ostream&,const Formula&);

#endif /* FORMULA_HPP_ */
