/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010 University of Bonn. All rights reserved.
 * Please see the LICENSE file or "Copyright notice" in builder.cpp for details.
 */

/*
 * ConfigFileBuffer.cpp
 *
 *  Created on: 12.06.2010
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "CodePatterns/MemDebug.hpp"

#include <iostream>
#include <boost/tokenizer.hpp>
#include <string>

#include "ConfigFileBuffer.hpp"
#include "CodePatterns/Verbose.hpp"
#include "CodePatterns/Log.hpp"
#include "Helpers/defs.hpp"
#include "Helpers/helpers.hpp"
#include "World.hpp"

/******************************** Functions for class ConfigFileBuffer **********************/

/** Structure containing compare function for Ion_Type sorting.
 */
struct IonTypeCompare {
  bool operator()(std::string s1, std::string s2) const {
    ConvertTo<int> toInt;
    boost::char_separator<char> sep("_");
    tokenizer tokens1(s1,sep);
    tokenizer tokens2(s2,sep);
    tokenizer::iterator tok_iter1 = tokens1.begin();
    tokenizer::iterator tok_iter2 = tokens2.begin();
    ++tok_iter1;
    ++tok_iter2;

    std::string element1(*tok_iter1++);
    std::string element2(*tok_iter2++);
    int elementno1 = toInt(element1.substr(4,string::npos));
    int elementno2 = toInt(element2.substr(4,string::npos));
    if (elementno1 != elementno2)
      return elementno1 < elementno2;
    else {
      std::string atom1(*tok_iter1);
      std::string atom2(*tok_iter2);
      int atomno1 = toInt(atom1);
      int atomno2 = toInt(atom2);
      return atomno1 < atomno2;
    }

//    char number1[8];
//    char number2[8];
//    const char *dummy1 = s1.c_str();
//    const char *dummy2 = s2.c_str();
//    //Log() << Verbose(0) << s1 << "  " << s2 << endl;
//    dummy1 = strchr(s1, '_')+sizeof(char)*5;  // go just after "Ion_Type"
//    dummy2 = strchr(dummy1, '_');
//    strncpy(number1, dummy1, dummy2-dummy1); // copy the number
//    number1[dummy2-dummy1]='\0';
//    dummy1 = strchr(s2, '_')+sizeof(char)*5;  // go just after "Ion_Type"
//    dummy2 = strchr(dummy1, '_');
//    strncpy(number2, dummy1, dummy2-dummy1); // copy the number
//    number2[dummy2-dummy1]='\0';
//    if (atoi(number1) != atoi(number2))
//      return (atoi(number1) < atoi(number2));
//    else {
//      dummy1 = strchr(s1, '_')+sizeof(char);
//      dummy1 = strchr(dummy1, '_')+sizeof(char);
//      dummy2 = strchr(dummy1, ' ') < strchr(dummy1, '\t') ? strchr(dummy1, ' ') : strchr(dummy1, '\t');
//      strncpy(number1, dummy1, dummy2-dummy1); // copy the number
//      number1[dummy2-dummy1]='\0';
//      dummy1 = strchr(s2, '_')+sizeof(char);
//      dummy1 = strchr(dummy1, '_')+sizeof(char);
//      dummy2 = strchr(dummy1, ' ') < strchr(dummy1, '\t') ? strchr(dummy1, ' ') : strchr(dummy1, '\t');
//      strncpy(number2, dummy1, dummy2-dummy1); // copy the number
//      number2[dummy2-dummy1]='\0';
//      return (atoi(number1) < atoi(number2));
//    }
  }

  typedef boost::tokenizer<boost::char_separator<char> > tokenizer;
};


/** Constructor for ConfigFileBuffer class.
 */
ConfigFileBuffer::ConfigFileBuffer() :
    buffer(NULL),
    LineMapping(NULL),
    CurrentLine(0),
    NoLines(0)
{
};

/** Constructor for ConfigFileBuffer class with filename to be parsed.
 * \param *filename file name
 */
ConfigFileBuffer::ConfigFileBuffer(const char * const filename) :
    buffer(NULL),
    LineMapping(NULL),
    CurrentLine(0),
    NoLines(0)
{
  InitFileBuffer(filename);
}

void ConfigFileBuffer::InitFileBuffer(const char * const filename)
{
  ifstream *file= new ifstream(filename);
  InitFileBuffer(file);
}

void ConfigFileBuffer::InitFileBuffer(istream *file)
{
  char line[MAXSTRINGSIZE];

  RemoveMapping();

  // prescan number of lines
  if (file->fail()) {
    DoeLog(1) && (eLog()<< Verbose(1) << "config file missing!" << endl);
    return;
  }
  NoLines = 0; // we're overcounting by one
  long file_position = file->tellg(); // mark current position
  do {
    file->getline(line, MAXSTRINGSIZE-1);
    NoLines++;
  } while (!file->eof());
  file->clear();
  file->seekg(file_position, ios::beg);
  DoLog(1) && (Log() << Verbose(1) << NoLines-1 << " lines were recognized." << endl);

  // allocate buffer's 1st dimension
  if (buffer != NULL) {
    DoeLog(1) && (eLog()<< Verbose(1) << "FileBuffer->buffer is not NULL!" << endl);
    return;
  } else
    buffer = new char *[NoLines];

  // scan each line and put into buffer
  int lines=0;
  int i;
  do {
    buffer[lines] = new char[MAXSTRINGSIZE];
    file->getline(buffer[lines], MAXSTRINGSIZE-1);
    i = strlen(buffer[lines]);
    buffer[lines][i] = '\n';
    buffer[lines][i+1] = '\0';
    lines++;
  } while((!file->eof()) && (lines < NoLines));
  DoLog(1) && (Log() << Verbose(1) << lines-1 << " lines were read into the buffer." << endl);
  file->clear();
  file->seekg(file_position, ios::beg);

  InitMapping();
}

/** Destructor for ConfigFileBuffer class.
 */
ConfigFileBuffer::~ConfigFileBuffer()
{
  RemoveBuffer();
  RemoveMapping();
}


/** Create trivial mapping.
 */
void ConfigFileBuffer::InitMapping()
{
  LineMapping = new int[NoLines];
  for (int i=0;i<NoLines;i++)
    LineMapping[i] = i;
  MappingAllocated = true;
}

/** Remove allocated mapping.
 */
void ConfigFileBuffer::RemoveMapping()
{
  delete[](LineMapping);
  MappingAllocated = false;
}

/** Remove allocated mapping.
 */
void ConfigFileBuffer::RemoveBuffer()
{
  for(int i=0;i<NoLines;++i)
    delete[](buffer[i]);
  delete[](buffer);
}


/** Creates a mapping for the \a *FileBuffer's lines containing the Ion_Type keyword such that they are sorted.
 * \a *map on return contains a list of NoAtom entries such that going through the list, yields indices to the
 * lines in \a *FileBuffer in a sorted manner of the Ion_Type?_? keywords. We assume that ConfigFileBuffer::CurrentLine
 * points to first Ion_Type entry.
 * \param *FileBuffer pointer to buffer structure
 * \param NoAtoms of subsequent lines to look at
 */
void ConfigFileBuffer::MapIonTypesInBuffer(const int NoAtoms)
{
  std::multimap<std::string, int, IonTypeCompare> IonTypeLineMap;
  if (!MappingAllocated) {
    InitMapping();
  }

  typedef boost::tokenizer<boost::char_separator<char> >
      tokenizer;
  boost::char_separator<char> sep("\t ");

  // put all into hashed map
  for (int i=CurrentLine; i<NoLines; ++i) {
    std::string line(buffer[i]);
    tokenizer tokens(line, sep);
    if (tokens.begin() != tokens.end()) {
      const std::string token = *tokens.begin();
      if (token.find("Ion_Type") != string::npos) {
        IonTypeLineMap.insert(pair<std::string, int> (token, i));
      }
    }
  }

  // fill map (aka IonType1_1, IonType1_1, IonType1_1, IonType1_2, IonType1_2, IonType1_2, ...
  // ..., IonType2_1, IonType2_1, IonType2_1, ...)
  int nr=0;
  for (map<std::string, int, IonTypeCompare>::iterator runner = IonTypeLineMap.begin(); runner != IonTypeLineMap.end(); ++runner) {
    if (CurrentLine+nr < NoLines)
      LineMapping[CurrentLine+(nr++)] = runner->second;
    else {
      DoeLog(0) && (eLog()<< Verbose(0) << "config::MapIonTypesInBuffer - NoLines is wrong: We are past the end of the file!" << endl);
      performCriticalExit();
    }
  }
}
