/*
 * Project: JobMarket
 * Description: asynchronous Server/Controller/Client-approach to parallel computing, based on boost::asio
 * Copyright (C)  2012 Frederik Heber. All rights reserved.
 * 
 */

/*
 * GetNextJobIdOperation.cpp
 *
 *  Created on: Feb 16, 2012
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

// boost asio needs specific operator new
#include <boost/asio.hpp>

//#include "CodePatterns/MemDebug.hpp"

#include "JobMarket/Operations/Controllers/GetNextJobIdOperation.hpp"

#include <boost/bind.hpp>
#include <iostream>
#include "JobMarket/Connection.hpp" // Must come before boost/serialization headers.
#include <boost/serialization/vector.hpp>
#include "CodePatterns/Info.hpp"
#include "CodePatterns/Log.hpp"
#include "JobMarket/ControllerChoices.hpp"
#include "JobMarket/types.hpp"

/** Handle connect operation to send number of done jobs.
 *
 * \param e error code if something went wrong
 * \param endpoint_iterator endpoint of the connection
 */
void GetNextJobIdOperation::handle_connect(const boost::system::error_code& e,
    boost::asio::ip::tcp::resolver::iterator endpoint_iterator)
{
  DEBUG_FUNCTION_ENTRYEXIT
  if (!e)
  {
    // Successfully established connection. Give choice.
    enum ControllerChoices choice = GetNextJobId;
    connection_.async_write(choice,
      boost::bind(&GetNextJobIdOperation::handle_SendNumberIds, this,
      boost::asio::placeholders::error));
  } else if (endpoint_iterator != boost::asio::ip::tcp::resolver::iterator()) {
    // Try the next endpoint.
    connection_.socket().close();
    boost::asio::ip::tcp::endpoint endpoint = *endpoint_iterator;
    connection_.socket().async_connect(endpoint,
      boost::bind(&GetNextJobIdOperation::handle_connect, this,
      boost::asio::placeholders::error, ++endpoint_iterator));
  } else {
    // An error occurred. Log it and return. Since we are not starting a new
    // operation the io_service will run out of work to do and the client will
    // exit.
    ELOG(1, e.message());
    AsyncOperation::handle_FinishOperation(e);
  }
}

/** Callback function when sending number of desired job ids.
 *
 * \param e error code if something went wrong
 */
void GetNextJobIdOperation::handle_SendNumberIds(const boost::system::error_code& e)
{
  DEBUG_FUNCTION_ENTRYEXIT
  if (!e)
  {
    // The connection::async_read() function will automatically
    // decode the data that is written to the underlying socket.
    LOG(1, "INFO: Requesting " << NumberIds <<  " ids  ...");
    connection_.async_write(NumberIds,
      boost::bind(&GetNextJobIdOperation::handle_ReceiveNextIds, this,
      boost::asio::placeholders::error));
  }
  else
  {
    // An error occurred.
    ELOG(1, e.message());
    AsyncOperation::handle_FinishOperation(e);
  }
}

/** Callback function when doneJobs have been received.
 *
 * \param e error code if something went wrong
 */
void GetNextJobIdOperation::handle_ReceiveNextIds(const boost::system::error_code& e)
{
  DEBUG_FUNCTION_ENTRYEXIT
  if (!e)
  {
    // The connection::async_read() function will automatically
    // decode the data that is written to the underlying socket.
    LOG(1, "INFO: Obtaining a bunch of available job ids ...");
    ids.clear();
    connection_.async_read(ids,
      boost::bind(&GetNextJobIdOperation::handle_FinishOperation, this,
      boost::asio::placeholders::error));
  }
  else
  {
    // An error occurred.
    ELOG(1, e.message());
    AsyncOperation::handle_FinishOperation(e);
  }
}

/** Callback function when an operation has been completed.
 *
 * \param e error code if something went wrong
 */
void GetNextJobIdOperation::handle_FinishOperation(const boost::system::error_code& e)
{
  DEBUG_FUNCTION_ENTRYEXIT

  // place id into nextids
  LOG(1, "INFO: Received " << ids.size() << " job ids.");
  nextids.insert(nextids.end(), ids.begin(), ids.end());
  ids.clear();
  NumberIds = 0;

  // and finish operation
  AsyncOperation::handle_FinishOperation(e);
}

/** Getter for nextid.
 *
 * \sa checkResults()
 * \param next available JobId_t
 */
size_t GetNextJobIdOperation::getNextId()
{
  ASSERT( nextids.size() != (size_t)0,
      "GetNextJobIdOperation::getNextId() - no next id available.");
  const JobId_t availableid = nextids.front();
  nextids.pop_front();
  return availableid;
}
