/*
 * Project: JobMarket
 * Description: asynchronous Server/Controller/Client-approach to parallel computing, based on boost::asio
 * Copyright (C)  2011-2012 Frederik Heber. All rights reserved.
 * 
 */

/*
 * CheckResultsOperation.cpp
 *
 *  Created on: Dec 11, 2011
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

// boost asio needs specific operator new
#include <boost/asio.hpp>

//#include "CodePatterns/MemDebug.hpp"

#include "JobMarket/Operations/Controllers/CheckResultsOperation.hpp"

#include <boost/bind.hpp>
#include <iostream>
#include "JobMarket/Connection.hpp" // Must come before boost/serialization headers.
#include <boost/serialization/set.hpp>
#include <boost/serialization/vector.hpp>
#include "CodePatterns/Info.hpp"
#include "CodePatterns/Log.hpp"
#include "JobMarket/ControllerChoices.hpp"

// static entities
const std::set<JobId_t> CheckResultsOperation::emptyids;

/** Handle connect operation to send number of done jobs.
 *
 * \param e error code if something went wrong
 * \param endpoint_iterator endpoint of the connection
 */
void CheckResultsOperation::handle_connect(const boost::system::error_code& e,
    boost::asio::ip::tcp::resolver::iterator endpoint_iterator)
{
  DEBUG_FUNCTION_ENTRYEXIT
  if (!e)
  {
    // Successfully established connection. Give choice.
    enum ControllerChoices choice = CheckState;
    connection_.async_write(choice,
      boost::bind(&CheckResultsOperation::handle_SendJobIds, this,
      boost::asio::placeholders::error));
  } else if (endpoint_iterator != boost::asio::ip::tcp::resolver::iterator()) {
    // Try the next endpoint.
    connection_.socket().close();
    boost::asio::ip::tcp::endpoint endpoint = *endpoint_iterator;
    connection_.socket().async_connect(endpoint,
      boost::bind(&CheckResultsOperation::handle_connect, this,
      boost::asio::placeholders::error, ++endpoint_iterator));
  } else {
    // An error occurred. Log it and return. Since we are not starting a new
    // operation the io_service will run out of work to do and the client will
    // exit.
    ELOG(1, e.message());
    AsyncOperation::handle_FinishOperation(e);
  }
}

/** Callback function when doneJobs have been received.
 *
 * \param e error code if something went wrong
 */
void CheckResultsOperation::handle_SendJobIds(const boost::system::error_code& e)
{
  DEBUG_FUNCTION_ENTRYEXIT
  if (!e)
  {
    // The connection::async_write() function will automatically
    // decode the data that is written to the underlying socket.
    LOG(1, "INFO: Sending vector of desired " << jobids.size() << " jobids ...");
    connection_.async_write(jobids,
      boost::bind(&CheckResultsOperation::handle_ReceiveJobInfo, this,
      boost::asio::placeholders::error));
  }
  else
  {
    // An error occurred.
    ELOG(1, e.message());
    AsyncOperation::handle_FinishOperation(e);
  }
}

/** Callback function when doneJobs have been received.
 *
 * \param e error code if something went wrong
 */
void CheckResultsOperation::handle_ReceiveJobInfo(const boost::system::error_code& e)
{
  DEBUG_FUNCTION_ENTRYEXIT
  if (!e)
  {
    // The connection::async_read() function will automatically
    // decode the data that is written to the underlying socket.
    LOG(1, "INFO: Obtaining number of present and done jobs ...");
    jobInfo.clear();
    connection_.async_read(jobInfo,
      boost::bind(&CheckResultsOperation::handle_FinishOperation, this,
      boost::asio::placeholders::error));
  }
  else
  {
    // An error occurred.
    ELOG(1, e.message());
    AsyncOperation::handle_FinishOperation(e);
  }
}

/** Getter for jobInfo.
 *
 * \sa checkResults()
 * \param number of present jobs
 */
size_t CheckResultsOperation::getPresentJobs() const
{
  ASSERT( jobInfo.size() == (size_t)2,
      "CheckResultsOperation::getPresentJobs() - jobInfo does not contain present and done jobs.");
  return jobInfo[0];
}

/** Getter for jobInfo.
 *
 * \sa checkResults()
 * \param number of done jobs
 */
size_t CheckResultsOperation::getDoneJobs() const
{
  ASSERT( jobInfo.size() == (size_t)2,
      "CheckResultsOperation::getPresentJobs() - jobInfo does not contain present and done jobs.");
  return jobInfo[1];
}

