/*
 * Project: JobMarket
 * Description: asynchronous Server/Controller/Client-approach to parallel computing, based on boost::asio
 * Copyright (C)  2012 Frederik Heber. All rights reserved.
 * 
 */

/*
 * FragmentScheduler_ControllerListener.cpp
 *
 *  Created on: Sep 5, 2012
 *      Author: heber
 */


// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

// boost asio needs specific operator new
#include <boost/asio.hpp>

//#include "CodePatterns/MemDebug.hpp"

#include <algorithm>
#include <boost/bind.hpp>
#include <sstream>
#include <string>
#include <vector>

#include "JobMarket/Connection.hpp" // Must come before boost/serialization headers.

#include <boost/serialization/set.hpp>
#include <boost/serialization/vector.hpp>

#include "CodePatterns/Info.hpp"
#include "CodePatterns/Log.hpp"

#include "CodePatterns/Observer/Notification.hpp"
#include "JobMarket/ControllerChoices.hpp"
#include "JobMarket/JobId.hpp"

#include "JobMarket/FragmentScheduler.hpp"

/** Handle a new controller connection.
 *
 * \sa handle_ReceiveJobs()
 * \sa handle_CheckResultState()
 * \sa handle_SendResults()
 *
 * \param e error code if something went wrong
 * \param conn reference with the connection
 */
void FragmentScheduler::ControllerListener_t::handle_Accept(const boost::system::error_code& e, connection_ptr conn)
{
  DEBUG_FUNCTION_ENTRYEXIT
  if (!e)
  {
    conn->async_read(choice,
      boost::bind(&FragmentScheduler::ControllerListener_t::handle_ReadChoice, this,
      boost::asio::placeholders::error, conn));
  }
  else
  {
    // An error occurred. Log it and return. Since we are not starting a new
    // accept operation the io_service will run out of work to do and the
    // server will exit.
    setExitflag( ErrorFlag );
    ELOG(0, e.message());
  }
}

/** Controller callback function to read the choice for next operation.
 *
 * \param e error code if something went wrong
 * \param conn reference with the connection
 */
void FragmentScheduler::ControllerListener_t::handle_ReadChoice(const boost::system::error_code& e, connection_ptr conn)
{
  DEBUG_FUNCTION_ENTRYEXIT
  if (!e)
  {
    bool LaunchNewAcceptor = true;
    LOG(1, "INFO: Received request for operation " << choice << ".");
    // switch over the desired choice read previously
    switch(choice) {
    case NoControllerOperation:
    {
      ELOG(1, "ControllerListener_t::handle_ReadChoice() - called with NoOperation.");
      break;
    }
    case GetNextJobId:
    {
      LOG(1, "INFO: Receiving number of desired job ids from controller ...");
      NumberIds = 0;
      conn->async_read(NumberIds,
        boost::bind(&FragmentScheduler::ControllerListener_t::handle_GetNextJobIdState, this,
        boost::asio::placeholders::error, conn));
      break;
    }
    case SendJobs:
    {
      // The connection::async_write() function will automatically
      // serialize the data structure for us.
      LOG(1, "INFO: Receiving bunch of jobs from a controller ...");
      jobs.clear();
      conn->async_read(jobs,
        boost::bind(&FragmentScheduler::ControllerListener_t::handle_ReceiveJobs, this,
        boost::asio::placeholders::error, conn));
      break;
    }
    case CheckState:
    {
      // get vector of jobids
      LOG(1, "INFO: Receiving bunch of jobids from a controller ...");
      jobids.clear();
      conn->async_read(jobids,
        boost::bind(&FragmentScheduler::ControllerListener_t::handle_CheckResultState, this,
        boost::asio::placeholders::error, conn));
      break;
    }
    case EmptyResultQueue:
    {
      LOG(1, "INFO: Removing all results in the Result queue ...");
      JobsQueue.removeWaitingResults();
      LOG(2, "DEBUG: Queue has " << JobsQueue.getDoneJobs() << " results.");
      break;
    }
    case EmptyJobQueue:
    {
      LOG(1, "INFO: Removing all jobs in the jobs queue ...");
      JobsQueue.removeWaitingJobs();
      LOG(2, "DEBUG: Queue has " << JobsQueue.getPresentJobs() << " results.");
      break;
    }
    case RemoveAll:
    {
      removeallWorkers();
      break;
    }
    case ReceiveResults:
    {
      // get vector of jobids
      LOG(1, "INFO: Receiving bunch of jobids from a controller ...");
      jobids.clear();
      conn->async_read(jobids,
        boost::bind(&FragmentScheduler::ControllerListener_t::handle_SendResults, this,
        boost::asio::placeholders::error, conn));
      break;
    }
    case ShutdownControllerSocket:
    {
      LOG(1, "INFO: Received shutdown from controller ...");
      // warn on shutdown if there are still jobs in the queue
      if (!JobsQueue.isJobPresent()) {
        // we shutdown? Hence, also shutdown controller
        LaunchNewAcceptor = !shutdownAllSockets();
      } else {
        ELOG(2, "There are still jobs waiting in the queue.");
      }
      break;
    }
    default:
      setExitflag( ErrorFlag );
      ELOG(1, "ControllerListener_t::handle_ReadChoice() - called with no valid choice.");
      break;
    }
    // restore NoControllerOperation choice such that choice is not read twice
    choice = NoControllerOperation;

    if (LaunchNewAcceptor) {
      LOG(1, "Launching new acceptor on socket.");
      // Start an accept operation for a new Connection.
      initiateSocket();
    }
  }
  else
  {
    // An error occurred. Log it and return. Since we are not starting a new
    // accept operation the io_service will run out of work to do and the
    // server will exit.
    setExitflag( ErrorFlag );
    ELOG(0, e.message());
  }
}

/** Controller callback function when job has been sent.
 *
 * We check here whether the worker socket is accepting, if there
 * have been no jobs we re-activate it, as it is shut down after
 * last job.
 *
 * \param e error code if something went wrong
 * \param conn reference with the connection
 */
void FragmentScheduler::ControllerListener_t::handle_ReceiveJobs(const boost::system::error_code& e, connection_ptr conn)
{
  DEBUG_FUNCTION_ENTRYEXIT
  // jobs are received, hence place in JobsQueue
  if (!jobs.empty()) {
    LOG(1, "INFO: Pushing " << jobs.size() << " jobs into queue.");
    if (!JobsQueue.pushJobs(jobs))
      ELOG(1, "The vector of received jobs contains illegal or present ids.");
  } else {
    ELOG(1, "The vector of received jobs is empty.");
  }
  jobs.clear();
}

/** Controller callback function when checking on state of results.
 *
 * \param e error code if something went wrong
 * \param conn reference with the connection
 */
void FragmentScheduler::ControllerListener_t::handle_CheckResultState(const boost::system::error_code& e, connection_ptr conn)
{
  DEBUG_FUNCTION_ENTRYEXIT
  // first update number
  jobInfo[0] = JobsQueue.getPresentJobs();
  if (jobids.empty())
    jobInfo[1] = JobsQueue.getDoneJobs( );
  else
    jobInfo[1] = JobsQueue.getDoneJobs( jobids );
  // now we accept connections to check for state of calculations
  LOG(1, "INFO: Sending state that "+toString(jobInfo[0])+" jobs are present and "+toString(jobInfo[1])+" jobs are done to controller ...");
  conn->async_write(jobInfo,
    boost::bind(&FragmentScheduler::ControllerListener_t::handle_finishCheckResultState, this,
    boost::asio::placeholders::error, conn));
  // do nothing
  LOG(1, "INFO: Sent that " << jobInfo << " jobs are (scheduled, done).");
}

/** Helper function to print a bunch of job ids while preserving space.
 *
 * @param nextids vector of ids to print
 * @return string such as "1-7,9,11-15", i.e. compressing to ranges
 */
std::string EfficientJobIdPrinter(const std::vector<JobId_t> &nextids)
{
  std::stringstream stream;
  if (!nextids.empty()) {
    std::vector<JobId_t>::const_iterator iter = nextids.begin();
    std::vector<JobId_t>::const_iterator advanceiter = iter;
    bool firstflag = true;
    JobId_t markstart = FragmentJob::IllegalJob;
    while (advanceiter != nextids.end()) {
      if (*advanceiter - 1 == *iter) {
        // next number is adjacent, i.e. we are in a range
        ++advanceiter;
        ++iter;
      } else if (advanceiter == iter) {
        // we are at the start of range, both are on top of each other
        if (firstflag)
          firstflag = false;
        else
          stream << ",";
        stream << *iter;
        markstart = *iter;
        ++advanceiter;
      } else {
        // end this range, set both on top of start of next range
        if (markstart != *iter)
          stream << "-" << *iter;
        iter = advanceiter;
      }
    }
    // print last number
    if (markstart != *iter)
      stream << "-" << *iter;
  }
  return stream.str();
}

/** Controller callback function when checking on state of results.
 *
 * \param e error code if something went wrong
 * \param conn reference with the connection
 */
void FragmentScheduler::ControllerListener_t::handle_finishCheckResultState(const boost::system::error_code& e, connection_ptr conn)
{
  // do nothing
  LOG(1, "INFO: Sent that " << jobInfo << " jobs are (scheduled, done).");
}


/** Controller callback function when checking on state of results.
 *
 * \param e error code if something went wrong
 * \param conn reference with the connection
 */
void FragmentScheduler::ControllerListener_t::handle_GetNextJobIdState(const boost::system::error_code& e, connection_ptr conn)
{
  DEBUG_FUNCTION_ENTRYEXIT

  std::vector<JobId_t> nextids( NumberIds, JobId::IllegalJob);
  std::generate(nextids.begin(), nextids.end(),
      boost::bind(&GlobalJobId::getNextId, boost::ref(globalId)));
  LOG(1, "INFO: Sending next available job ids " << EfficientJobIdPrinter(nextids) << " to controller ...");
  conn->async_write(nextids,
    boost::bind(&FragmentScheduler::ControllerListener_t::handle_SendIds, this,
    boost::asio::placeholders::error, conn));
}

/** Controller callback function when free job ids have been sent.
 *
 * \param e error code if something went wrong
 * \param conn reference with the connection
 */
void FragmentScheduler::ControllerListener_t::handle_SendIds(const boost::system::error_code& e, connection_ptr conn)
{
  DEBUG_FUNCTION_ENTRYEXIT
  // do nothing
  LOG(1, "INFO: Ids have been sent.");
}

/** Controller callback function when result has been received.
 *
 * \param e error code if something went wrong
 * \param conn reference with the connection
 */
void FragmentScheduler::ControllerListener_t::handle_SendResults(const boost::system::error_code& e, connection_ptr conn)
{
  DEBUG_FUNCTION_ENTRYEXIT

  const std::vector<FragmentResult::ptr> results =
      jobids.empty() ?
          std::vector<FragmentResult::ptr>() :
          JobsQueue.getAllResults(jobids);
  // ... or we give the results
  LOG(1, "INFO: Sending "+toString(results.size())+" results to controller ...");
  conn->async_write(results,
    boost::bind(&FragmentScheduler::ControllerListener_t::handle_finishSendResults, this,
    boost::asio::placeholders::error, conn));
}

/** Controller callback function when result has been received.
 *
 * \param e error code if something went wrong
 * \param conn reference with the connection
 */
void FragmentScheduler::ControllerListener_t::handle_finishSendResults(const boost::system::error_code& e, connection_ptr conn)
{
  DEBUG_FUNCTION_ENTRYEXIT
  // do nothing
  LOG(1, "INFO: Results have been sent.");
}
