/*
 * FragmentScheduler.hpp
 *
 *  Created on: Oct 19, 2011
 *      Author: heber
 */

#ifndef FRAGMENTSCHEDULER_HPP_
#define FRAGMENTSCHEDULER_HPP_

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <boost/asio.hpp>
#include <boost/function.hpp>
#include <boost/shared_ptr.hpp>
#include <deque>
#include <set>
#include <vector>

#include "CodePatterns/Observer/Observer.hpp"
#include "JobMarket/Connection.hpp"
#include "JobMarket/ControllerChoices.hpp"
#include "JobMarket/Operations/AsyncOperation.hpp"
#include "JobMarket/Operations/OperationQueue.hpp"
#include "Operations/Servers/ShutdownWorkerOperation.hpp"
#include "JobMarket/ControllerChoices.hpp"
#include "JobMarket/FragmentQueue.hpp"
#include "JobMarket/GlobalJobId.hpp"
#include "JobMarket/Jobs/FragmentJob.hpp"
#include "JobMarket/Listener.hpp"
#include "JobMarket/Results/FragmentResult.hpp"
#include "JobMarket/types.hpp"
#include "JobMarket/Pool/PoolGuard.hpp"
#include "JobMarket/Pool/WorkerPool.hpp"
#include "JobMarket/WorkerAddress.hpp"
#include "JobMarket/WorkerChoices.hpp"

/** FragmentScheduler serves FragmentJobs to Workers and accepts commands from
 * a Controller.
 *
 */
class FragmentScheduler : public Observer
{
public:
  /// Constructor opens the acceptor and starts waiting for the first incoming
  /// Connection.
  FragmentScheduler(
      boost::asio::io_service& io_service,
      unsigned short workerport,
      unsigned short controllerport,
      const size_t timeout
      );
  ~FragmentScheduler();

  void shutdown(int sig);
private:
  void sendJobToWorker(const WorkerAddress &address, FragmentJob::ptr &job);
  void sendAvailableJobToNextIdleWorker();
  bool shutdown();
  void shutdownWorker(const WorkerAddress &address);
  void removeAllWorkers();
  void removeWorker(const WorkerAddress address);
  void unmarkWorkerBusy(const WorkerAddress address);
  void cleanupOperationQueue(AsyncOperation *op);
  void workerAcceptsJob(const WorkerAddress _address, const JobId_t _id);
  void workerRejectsJob(const WorkerAddress _address, const JobId_t _id);

  void update(Observable *publisher);
  void recieveNotification(Observable *publisher, Notification_ptr notification);
  void subjectKilled(Observable *publisher);

  class WorkerListener_t : public Listener
  {
  public:
    WorkerListener_t(
        boost::asio::io_service& io_service,
        unsigned short port,
        FragmentQueue &_JobsQueue,
        WorkerPool &_pool,
        boost::function<void (const WorkerAddress&, FragmentJob::ptr&)> _callback,
        boost::function<void (const WorkerAddress)> _unmarkBusyWorkerFunction) :
      Listener(io_service, port),
      JobsQueue(_JobsQueue),
      pool(_pool),
      callback_sendJobToWorker(_callback),
      unmarkBusyWorkerFunction(_unmarkBusyWorkerFunction)
    {}
    virtual ~WorkerListener_t() {}

  private:

    /** This class contains all data that are unique to a specific chain
     * of handlers.
     *
     * This way a client that connects to send back his result does not
     * overwrite the address of a previously connecting client that enrolls.
     *
     * This class should be used wrapped inside a shared_ptr to avoid having
     * to worry about memory management. The instance is created by
     * handle_Accept() and handed as pointer to each handler in the chain.
     */
    struct HandlerData {
      /** Default Constructor for Handler data.
       *
       */
      HandlerData();

      //!> typedef for usage of class instances inside shared pointer.
      typedef boost::shared_ptr<HandlerData> ptr;

      //!> address of new Worker
      WorkerAddress address;

      //!> result that is received from the client.
      FragmentResult::ptr result;

      //!> choice
      enum WorkerChoices choice;
    };

  protected:
    /// Handle completion of a accept worker operation.
    void handle_Accept(const boost::system::error_code& e, connection_ptr conn);

    /// Handle completion of Worker operation to read choice
    void handle_ReadChoice(const boost::system::error_code& e, connection_ptr conn, HandlerData::ptr &data);

    /// Worker callback function when job has been sent.
    void handle_ReadAddress(const boost::system::error_code& e, connection_ptr conn, HandlerData::ptr &data);

    /// Worker callback function when new worker has enrolled.
    void handle_enrolled(const boost::system::error_code& e, connection_ptr conn, HandlerData::ptr &data);

    /// Worker callback function when worker has been informed of successful removal.
    void handle_removed(const boost::system::error_code& e, connection_ptr conn, HandlerData::ptr &data);

    /// Worker callback function when result has been received.
    void handle_ReceiveResultFromWorker(const boost::system::error_code& e, connection_ptr conn, HandlerData::ptr &data);

    /// Worker callback function when invalid result has been received.
    void handle_RejectResultFromWorker(const boost::system::error_code& e, connection_ptr conn, HandlerData::ptr &data);

  private:
    //!> reference to Queue
    FragmentQueue &JobsQueue;

    //!> callback reference to container class
    WorkerPool &pool;

    //!> callback function to access send job function
    boost::function<void (const WorkerAddress&, FragmentJob::ptr&)> callback_sendJobToWorker;

    //!> function to call when a busy worker has finished and sent its result
    boost::function<void (const WorkerAddress)> unmarkBusyWorkerFunction;
  };

  class ControllerListener_t : public Listener
  {
  public:
    ControllerListener_t(
        boost::asio::io_service& io_service,
        unsigned short port,
        FragmentQueue &_JobsQueue,
        boost::function<void ()> _removeallWorkers,
        boost::function<bool ()> _shutdownAllSockets) :
      Listener(io_service, port),
      JobsQueue(_JobsQueue),
      jobInfo((size_t)2, 0),
      NumberIds(0),
      choice(NoControllerOperation),
      removeallWorkers(_removeallWorkers),
      shutdownAllSockets(_shutdownAllSockets),
      globalId(0)
    {}
    virtual ~ControllerListener_t() {}

  protected:
    /// Handle completion of a accept controller operation.
    void handle_Accept(const boost::system::error_code& e, connection_ptr conn);

    /// Handle completion of controller operation to read choice
    void handle_ReadChoice(const boost::system::error_code& e, connection_ptr conn);

    /// Controller callback function when job has been sent.
    void handle_ReceiveJobs(const boost::system::error_code& e, connection_ptr conn);

    /// Controller callback function when checking on state of results.
    void handle_CheckResultState(const boost::system::error_code& e, connection_ptr conn);

    /// Controller callback function when checking on state of results is done.
    void handle_finishCheckResultState(const boost::system::error_code& e, connection_ptr conn);

    /// Controller callback function when checking on state of results.
    void handle_GetNextJobIdState(const boost::system::error_code& e, connection_ptr conn);

    /// Controller callback function when free job ids have been sent.
    void handle_SendIds(const boost::system::error_code& e, connection_ptr conn);

    /// Controller callback function when result are to be sent.
    void handle_SendResults(const boost::system::error_code& e, connection_ptr conn);

    /// Controller callback function when results has been sent.
    void handle_finishSendResults(const boost::system::error_code& e, connection_ptr conn);

  private:
    //!> reference to external FragmentQueue containing jobs to work on
    FragmentQueue & JobsQueue;

    //!> bunch of jobs received from controller before placed in JobsQueue
    std::vector<FragmentJob::ptr> jobs;

    //!> number of jobs that are waiting to be and are calculated, required for returning status
    std::vector<size_t> jobInfo;

    //!> vector of jobids for requesting results
    std::set<JobId_t> jobids;

    //!> number of job ids request from controller;
    size_t NumberIds;

    //!> choice
    enum ControllerChoices choice;

    //!> bound function to remove all workers
    boost::function<void ()> removeallWorkers;

    //!> bound function to shutdown all sockets
    boost::function<bool ()> shutdownAllSockets;

    // TODO: replace this instance by a IdPool.
    //!> global id to give next available job id
    GlobalJobId globalId;
  };

private:
  //!> static entity to indicate to clients that the queue is empty.
  static FragmentJob::ptr NoJob;

  //!> reference to the io_service which we use for connections
  boost::asio::io_service& io_service;

  //!> Queue with data to be sent to each client.
  FragmentQueue JobsQueue;

  //!> Pool of Workers
  WorkerPool pool;

  //!> Listener instance that waits for a worker
  WorkerListener_t  WorkerListener;

  //!> Listener instance that waits for a controller
  ControllerListener_t  ControllerListener;

public:
  /** Getter for Exitflag.
   *
   * @return Exitflag of operations
   */
  size_t getExitflag() const
  {
    if (WorkerListener.getExitflag() != 0)
      return WorkerListener.getExitflag();
    if (ControllerListener.getExitflag() != 0)
      return ControllerListener.getExitflag();
    return 0;
  }

private:
  //!> Connection for sending jobs to workers
  Connection connection;

  //!> internal queue for all asynchronous operations
  OperationQueue OpQueue;

private:
  //!> Guard that checks if workers are unreachable
  PoolGuard guard;
};

#endif /* FRAGMENTSCHEDULER_HPP_ */
