/*
 * RotateAroundOriginByAngleAction.cpp
 *
 *  Created on: Aug 06, 2010
 *      Author: heber
 */

#include "Helpers/MemDebug.hpp"

#include "Actions/AtomAction/RotateAroundOriginByAngleAction.hpp"
#include "Actions/ActionRegistry.hpp"
#include "Helpers/Log.hpp"
#include "Helpers/Verbose.hpp"
#include "LinearAlgebra/Line.hpp"
#include "LinearAlgebra/Vector.hpp"
#include "molecule.hpp"


#include <iostream>
#include <fstream>
#include <string>

using namespace std;

#include "UIElements/UIFactory.hpp"
#include "UIElements/Dialog.hpp"
#include "Actions/ValueStorage.hpp"

/****** AtomRotateAroundOriginByAngleAction *****/

// memento to remember the state when undoing

class AtomRotateAroundOriginByAngleState : public ActionState {
public:
  AtomRotateAroundOriginByAngleState(std::vector<atom*> _selectedAtoms, const Vector &_Axis, const double _alpha) :
    selectedAtoms(_selectedAtoms),
    Axis(_Axis),
    alpha(_alpha)
  {}
  std::vector<atom*> selectedAtoms;
  Vector Axis;
  double alpha;
};

const char AtomRotateAroundOriginByAngleAction::NAME[] = "rotate-origin";

AtomRotateAroundOriginByAngleAction::AtomRotateAroundOriginByAngleAction() :
  Action(NAME)
{}

AtomRotateAroundOriginByAngleAction::~AtomRotateAroundOriginByAngleAction()
{}

void AtomRotateAroundOriginByAngle(const Vector &Axis, double angle) {
  ValueStorage::getInstance().setCurrentValue(AtomRotateAroundOriginByAngleAction::NAME, angle);
  ValueStorage::getInstance().setCurrentValue("position", Axis);
  ActionRegistry::getInstance().getActionByName(AtomRotateAroundOriginByAngleAction::NAME)->call(Action::NonInteractive);
};

Dialog* AtomRotateAroundOriginByAngleAction::fillDialog(Dialog *dialog) {
  ASSERT(dialog,"No Dialog given when filling action dialog");

  dialog->queryDouble(NAME, MapOfActions::getInstance().getDescription(NAME));
  dialog->queryVector("position", false, MapOfActions::getInstance().getDescription("position"));

  return dialog;
}

Action::state_ptr AtomRotateAroundOriginByAngleAction::performCall() {
  double alpha = 0.;
  Vector Axis;
  std::vector<atom *> selectedAtoms = World::getInstance().getSelectedAtoms();

  // obtain axis to rotate to
  ValueStorage::getInstance().queryCurrentValue(NAME, alpha);
  ValueStorage::getInstance().queryCurrentValue("position", Axis);

  // check whether Axis is valid
  if (Axis.IsZero())
    return Action::failure;

  // convert from degrees to radian
  alpha *= M_PI/180.;

  // Creation Line that is the rotation axis
  Line RotationAxis(Vector(0.,0.,0.), Axis);

  DoLog(0) && (Log() << Verbose(0) << "Rotate around origin by " << alpha << " radian, axis from origin to " << Axis << "." << endl);
  // TODO: use AtomSet::rotate?
  for (std::vector<atom *>::iterator iter = selectedAtoms.begin(); iter != selectedAtoms.end(); ++iter) {
    (*iter)->setPosition(RotationAxis.rotateVector((*iter)->getPosition(), alpha));
  }
  DoLog(0) && (Log() << Verbose(0) << "done." << endl);
  return Action::state_ptr(new AtomRotateAroundOriginByAngleState(World::getInstance().getSelectedAtoms(), Axis, alpha));
}

Action::state_ptr AtomRotateAroundOriginByAngleAction::performUndo(Action::state_ptr _state) {
  AtomRotateAroundOriginByAngleState *state = assert_cast<AtomRotateAroundOriginByAngleState*>(_state.get());

  // Creation Line that is the rotation axis
  Line RotationAxis(Vector(0.,0.,0.), state->Axis);

  for (std::vector<atom *>::iterator iter = state->selectedAtoms.begin(); iter != state->selectedAtoms.end(); ++iter) {
    (*iter)->setPosition(RotationAxis.rotateVector((*iter)->getPosition(), -state->alpha));
  }

  return Action::state_ptr(_state);
}

Action::state_ptr AtomRotateAroundOriginByAngleAction::performRedo(Action::state_ptr _state){
  AtomRotateAroundOriginByAngleState *state = assert_cast<AtomRotateAroundOriginByAngleState*>(_state.get());

  // Creation Line that is the rotation axis
  Line RotationAxis(Vector(0.,0.,0.), state->Axis);

  for (std::vector<atom *>::iterator iter = state->selectedAtoms.begin(); iter != state->selectedAtoms.end(); ++iter) {
    (*iter)->setPosition(RotationAxis.rotateVector((*iter)->getPosition(), state->alpha));
  }

  return Action::state_ptr(_state);
}

bool AtomRotateAroundOriginByAngleAction::canUndo() {
  return true;
}

bool AtomRotateAroundOriginByAngleAction::shouldUndo() {
  return true;
}

const string AtomRotateAroundOriginByAngleAction::getName() {
  return NAME;
}
