/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010-2012 University of Bonn. All rights reserved.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * BuildInducedSubgraph.cpp
 *
 *  Created on: Mar 3, 2011
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "CodePatterns/MemDebug.hpp"

#include "BuildInducedSubgraph.hpp"

#include "Atom/atom.hpp"
#include "Bond/bond.hpp"
#include "molecule.hpp"
#include "CodePatterns/Assert.hpp"
#include "CodePatterns/Log.hpp"
#include "CodePatterns/Verbose.hpp"


BuildInducedSubgraph::BuildInducedSubgraph(molecule * const _Son, const molecule * const _Father) :
  Son(_Son),
  Father(_Father)
{
  // reset parent list
  ParentList.clear();
}

BuildInducedSubgraph::~BuildInducedSubgraph()
{}


void BuildInducedSubgraph::FillParentList()
{
  // fill parent list with sons
  LOG(3, "Filling Parent List.");
  for (molecule::iterator iter = Son->begin(); iter != Son->end(); ++iter) {
    ParentList[(*iter)->father] = (*iter);
    // Outputting List for debugging
    LOG(4, "INFO: ParentList[] of " << (*iter)->father << " is " << *ParentList[(*iter)->father] << ".");
  }
}

bool BuildInducedSubgraph::CreateBondsFromParent()
{
  bool status = true;
  atom *OtherAtom = NULL;
  // check each entry of parent list and if ok (one-to-and-onto matching) create bonds
  LOG(2, "STATUS: Creating bonds.");
  for (molecule::iterator iter = Father->begin(); iter != Father->end(); ++iter) {
    if (ParentList.count(*iter)) {
      if (ParentList[(*iter)]->father != (*iter)) {
        status = false;
      } else {
        const BondList& ListOfBonds = (*iter)->getListOfBonds();
        for (BondList::const_iterator Runner = ListOfBonds.begin();
            Runner != ListOfBonds.end();
            ++Runner) {
          OtherAtom = (*Runner)->GetOtherAtom((*iter));
          if (ParentList.count(OtherAtom)) { // if otheratom is also a father of an atom on this molecule, create the bond
            LOG(4, "INFO: Endpoints of Bond " << (*Runner) << " are both present: " << ParentList[(*iter)]->getName() << " and " << ParentList[OtherAtom]->getName() << ".");
            Son->AddBond(ParentList[(*iter)], ParentList[OtherAtom], (*Runner)->getDegree());
          }
        }
      }
    }
  }
  return status;
}


bool BuildInducedSubgraph::operator()(){
  bool status = true;

  FillParentList();
  status = CreateBondsFromParent();

  return status;
}

