/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2010-2012 University of Bonn. All rights reserved.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * CreateAdjacencyAction.cpp
 *
 *  Created on: May 9, 2010
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "CodePatterns/MemDebug.hpp"

#include "CodePatterns/Assert.hpp"
#include "CodePatterns/Log.hpp"

#include "Descriptors/AtomSelectionDescriptor.hpp"
#include "Graph/BondGraph.hpp"
#include "molecule.hpp"
#include "World.hpp"

#include <iostream>
#include <list>
#include <string>

#include "Actions/GraphAction/CreateAdjacencyAction.hpp"

using namespace MoleCuilder;

// Storing undo state
struct BondInfo_t {
  atomId_t leftatom;
  atomId_t rightatom;
  size_t degree;
};

// and construct the stuff
#include "CreateAdjacencyAction.def"
#include "Action_impl_pre.hpp"
/** =========== define the function ====================== */
Action::state_ptr GraphCreateAdjacencyAction::performCall() {
  BondGraph *BG = World::getInstance().getBondGraph();
  ASSERT(BG != NULL, "GraphCreateAdjacencyAction: BondGraph is NULL.");

  // count all present bonds
  World::AtomComposite Set = World::getInstance().getAllAtoms(AtomsBySelection());
  std::vector<BondInfo_t> bonds;
  {
    size_t count_bonds = 0;
    for (World::AtomComposite::const_iterator iter = Set.begin();
        iter != Set.end();
        ++iter) {
      const atom * const Walker = *iter;
      count_bonds += Walker->getListOfBonds().size();
    }
    bonds.reserve(count_bonds/2);
  }
  // Storing undo info
  for (World::AtomComposite::const_iterator iter = Set.begin();
      iter != Set.end();
      ++iter) {
    const atom * const Walker = *iter;
    const BondList& ListOfBonds = Walker->getListOfBonds();
    for (BondList::const_iterator bonditer = ListOfBonds.begin();
        bonditer != ListOfBonds.end();
        ++bonditer) {
      const bond::ptr &CurrentBond = *bonditer;
      // if both atoms are in selected set, we check ids otherwise not
      if (((!World::getInstance().isSelected(CurrentBond->leftatom))
          || (!World::getInstance().isSelected(CurrentBond->rightatom)))
          || (CurrentBond->leftatom->getId() < CurrentBond->rightatom->getId())) {
        BondInfo_t BondInfo;
        BondInfo.leftatom = CurrentBond->leftatom->getId();
        BondInfo.rightatom = CurrentBond->rightatom->getId();
        BondInfo.degree = CurrentBond->getDegree();
        bonds.push_back(BondInfo);
      }
    }
  }
  GraphCreateAdjacencyState *UndoState = new GraphCreateAdjacencyState(bonds, params);

  // now recreate adjacency
  BG->CreateAdjacency(Set);

  // give info
  size_t BondCount = 0;
  std::vector<molecule *> molecules = World::getInstance().getAllMolecules();
  for (std::vector<molecule *>::const_iterator iter = molecules.begin();
      iter != molecules.end(); ++iter)
    BondCount += (*iter)->getBondCount();
  LOG(0, "STATUS: Recognized " << BondCount << " bonds.");

  return Action::state_ptr(UndoState);
}

Action::state_ptr GraphCreateAdjacencyAction::performUndo(Action::state_ptr _state) {
  GraphCreateAdjacencyState *state = assert_cast<GraphCreateAdjacencyState*>(_state.get());

  BondGraph *BG = World::getInstance().getBondGraph();
  ASSERT(BG != NULL, "GraphCreateAdjacencyAction: BondGraph is NULL.");

  // remove all bonds in the set
  World::AtomComposite Set = World::getInstance().getAllAtoms(AtomsBySelection());
  BG->cleanAdjacencyList(Set);

  // recreate from stored info
  const size_t CurrentTime = WorldTime::getTime();
  std::vector<BondInfo_t> &bonds = state->bonds;
  for(std::vector<BondInfo_t>::const_iterator iter = bonds.begin();
      iter != bonds.end(); ++iter) {
    atom * const Walker = World::getInstance().getAtom(AtomById(iter->leftatom));
    ASSERT( Walker != NULL,
        "GraphCreateAdjacencyAction::performUndo() - "+toString(iter->leftatom)+" missing.");
    atom * const OtherWalker = World::getInstance().getAtom(AtomById(iter->rightatom));
    ASSERT( OtherWalker != NULL,
        "GraphCreateAdjacencyAction::performUndo() - "+toString(iter->rightatom)+" missing.");
    bond::ptr CurrentBond = Walker->addBond(CurrentTime, OtherWalker);
    CurrentBond->setDegree(iter->degree);
  }
  return Action::state_ptr(_state);
}

Action::state_ptr GraphCreateAdjacencyAction::performRedo(Action::state_ptr _state){
  BondGraph *BG = World::getInstance().getBondGraph();
  ASSERT(BG != NULL, "GraphCreateAdjacencyAction: BondGraph is NULL.");

  // now recreate adjacency
  World::AtomComposite Set = World::getInstance().getAllAtoms(AtomsBySelection());
  BG->CreateAdjacency(Set);

  return Action::state_ptr(_state);
}

bool GraphCreateAdjacencyAction::canUndo() {
  return true;
}

bool GraphCreateAdjacencyAction::shouldUndo() {
  return true;
}
/** =========== end of function ====================== */
