/*
 * CompoundPotential.hpp
 *
 *  Created on: May 8, 2013
 *      Author: heber
 */

#ifndef COMPOUNDPOTENTIAL_HPP_
#define COMPOUNDPOTENTIAL_HPP_

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "FunctionApproximation/FunctionModel.hpp"
#include "Potentials/SerializablePotential.hpp"

#include <vector>

class CompoundPotentialTest;
class HomologyGraph;
class TrainingData;

/** CompoundPotential combines several EmipiricalPotential's into one
 * FunctionModel to allow for simultaneous FunctionApproximation to a single
 * fragment/graph.
 *
 * The CompoundPotential obtains a Graph as parameter to cstor and looks through
 * the PotentialRegistry marking all potentials whose ParticleTypes_t signature
 * matches. All of these are placed into an internal vector and used for
 * fitting.
 *
 * The main work is then in operator() and parameter_derivative(), where the
 * contained set of models has to be evaluated one after the other:
 *
 */
class CompoundPotential :
    public FunctionModel
{
  //!> grant unit test access to internal parts
  friend class CompoundPotentialTest;
  // some repeated typedefs to avoid ambiguities
  typedef FunctionModel::arguments_t arguments_t;
  typedef FunctionModel::result_t result_t;
  typedef FunctionModel::results_t results_t;
  typedef FunctionModel::parameters_t parameters_t;
public:
  /** Constructor for class CompoundPotential.
   *
   * \param graph graph to be checked against present ParticleTypes_t signatures
   *        of potentials in PotentialRegistry
   */
  CompoundPotential(const HomologyGraph &graph);

  /** Destructor for class CompoundPotential.
   *
   */
  virtual ~CompoundPotential();

  /** Setter for parameters as required by FunctionModel interface.
   *
   * \param _params given set of parameters
   */
  void setParameters(const parameters_t &_params);

  /** Getter for parameters as required by FunctionModel interface.
   *
   * \return set of parameters
   */
  parameters_t getParameters() const;

  /** Sets the parameter randomly within the sensible range of each parameter.
   *
   * \param data container with training data for guesstimating range
   */
  void setParametersToRandomInitialValues(const TrainingData &data);

  /** Getter for the number of parameters of this model function.
   *
   * \return number of parameters
   */
  size_t getParameterDimension() const;

  /** Sets the magic triple function that we use for getting angle distances.
   *
   * @param _triplefunction function that returns a list of triples (i.e. the
   *        two remaining distances) to a given pair of points (contained as
   *        indices within the argument)
   */
  void setTriplefunction(triplefunction_t &_triplefunction);

  /** Evaluates the harmonic potential function for the given arguments.
   *
   * @param arguments single distance
   * @return value of the potential function
   */
  results_t operator()(const arguments_t &arguments) const;

  /** Evaluates the derivative of the function with the given \a arguments
   * with respect to a specific parameter indicated by \a index.
   *
   * \param arguments set of arguments as input variables to the function
   * \param index derivative of which parameter
   * \return result vector containing the derivative with respect to the given
   *         input
   */
  results_t parameter_derivative(const arguments_t &arguments, const size_t index) const;

  /** States whether lower and upper boundaries should be used to constraint
   * the parameter search for this function model.
   *
   * \return true - constraints should be used, false - else
   */
  bool isBoxConstraint() const;

  /** Returns a vector which are the lower boundaries for each parameter_t
   * of this FunctionModel.
   *
   * \return vector of parameter_t resembling lowest allowed values
   */
  parameters_t getLowerBoxConstraints() const;

  /** Returns a vector which are the upper boundaries for each parameter_t
   * of this FunctionModel.
   *
   * \return vector of parameter_t resembling highest allowed values
   */
  parameters_t getUpperBoxConstraints() const;

  /** Returns a bound function to be used with TrainingData, extracting distances
   * from a Fragment.
   *
   * Here, we simply concatenate the arguments delivered by the extractors of each model.
   *
   * \return bound function extracting distances from a fragment
   */
  FunctionModel::extractor_t getSpecificExtractor() const;

  /** Returns a bound function to be used with TrainingData, extracting distances
   * from a Fragment.
   *
   * \note CompoundPotential has only default filter, as filter needs to be
   * model-specific.
   *
   * \return bound function extracting distances from a fragment
   */
  FunctionModel::filter_t getSpecificFilter() const;

  /** Returns the number of arguments the underlying function requires.
   *
   * We give the total sum over the argument counts of all models.
   *
   * \return number of arguments of the function
   */
  size_t getSpecificArgumentCount() const;

private:
  //!> typedef for split up arguments, each associated to a model
  typedef std::vector< std::pair<FunctionModel *, arguments_t> > arguments_by_model_t;

  /** Helper function to split up concatenated arguments for operator() calls.
   *
   * \param arguments arguments to split up
   * \return vector of partial arguments with associated model
   */
  arguments_by_model_t splitUpArgumentsByModels(const arguments_t &arguments) const;

  /** Helper function to split up total list of arguments for operator() calls.
   *
   * Here, we assume that we are given a list of all possible arguments and
   * each model has to filter out its share and align it into consecutive
   * bunches.
   *
   * \param arguments arguments to split up
   * \return vector of partial arguments with associated model
   */
  arguments_by_model_t splitUpArgumentsByModelsFilter(const arguments_t &arguments) const;

  /** Helper function to check whether split up argument bunch matches with types.
   *
   * \param types types of potential to check whether args match
   * \param args vector of argument whose types to check
   */
  bool areValidArguments(
      const SerializablePotential::ParticleTypes_t &types,
      const arguments_t &args) const;

public:
  //!> typedef for internal vector of simultaneously fitted models.
  typedef std::vector<FunctionModel*> models_t;

  models_t::const_iterator begin() const
  { return models.begin(); }

  models_t::const_iterator end() const
  { return models.end(); }

private:
  //!> Internal vector of FunctionModels to fit simultaneously.
  models_t models;

  //!> typedef for storing internally the ParticleTypes of each model.
  typedef std::vector<SerializablePotential::ParticleTypes_t> particletypes_per_model_t;
  //!> Internal vector of ParticleTypes for each model to use when constructing extractor
  particletypes_per_model_t particletypes_per_model;

  //!> static token of this potential type
  static const std::string potential_token;
};

#endif /* COMPOUNDPOTENTIAL_HPP_ */
