/*
 * QtObservedMolecule.hpp
 *
 *  Created on: Oct 28, 2015
 *      Author: heber
 */


#ifndef QTOBSERVEDMOLECULE_HPP_
#define QTOBSERVEDMOLECULE_HPP_

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <QtGui/QWidget>

#include <boost/function.hpp>
#include <boost/shared_ptr.hpp>

#include "CodePatterns/Observer/Observable.hpp"
#include "CodePatterns/Observer/Observer.hpp"

#include "molecule.hpp"
#include "UIElements/Qt4/InstanceBoard/ObservedValue_types.hpp"
#include "UIElements/Qt4/InstanceBoard/ObservedValuesContainer.hpp"
#include "types.hpp"

class QtObservedInstanceBoard;

/** This instance is the ObservedValue representation of a World's molecule.
 *
 * Due to the signal/slot mechanism and its delays, lifetime of objects in the
 * World and their QtGui representation cannot directly be related (without
 * slowing down Actions just for having the representation up to speed).
 * Hence, the required information for displaying and representing these
 * objects must be contained in an extra instance.
 *
 * This is the instance for information about a particular molecule.
 */
class QtObservedMolecule : public QWidget, public Observer
{
  Q_OBJECT

public:

  //!> typedef for instance wrapped in shared ptr
  typedef boost::shared_ptr<QtObservedMolecule> ptr;

private:
  //!> ObservedValuesContainer needs to access private cstor and dstor
  friend class ObservedValuesContainer<QtObservedMolecule, moleculeId_t>;
  //!> QtObservedInstanceBoard needs to access private cstor and dstor
  friend class QtObservedInstanceBoard;

  /** Cstor of QtObservedMolecule.
   *
   * \param _ObservedValues ref to set of observed values for this instance
   * \param _board ref to InstanceBoard for callbacks on occasion of subjectKilled()
   * \param _parent Qt parent to automatically destroy when parent is destroyed
   */
  QtObservedMolecule(
      const ObservedValues_t &_ObservedValues,
      QtObservedInstanceBoard &_board,
      QWidget * _parent=0);

public:

  /** Dstor of QtObservedMolecule.
   *
   */
  ~QtObservedMolecule();

  // Observer functions
  void update(Observable *publisher);
  void subjectKilled(Observable *publisher);
  void recieveNotification(Observable *publisher, Notification_ptr notification);

  /** Getter to molecule atom count contained in \a ObservedValues.
   *
   * \return molecule's atom count
   */
  int getAtomCount() const;

  /** Getter to molecule index contained in \a ObservedValues.
   *
   * \return molecule's index
   */
  moleculeId_t getMolIndex() const;

  /** Getter to molecule name contained in \a ObservedValues.
   *
   * \return molecule's name
   */
  std::string getMolName() const;

  /** Getter to molecule formula contained in \a ObservedValues.
   *
   * \return molecule's formula
   */
  std::string getMolFormula() const;

  /** Getter to molecule's bounding box contained in \a ObservedValues.
   *
   * \return molecule's bounding box
   */
  molecule::BoundingBoxInfo getBoundingBox() const;

  static const molecule * const getMolecule(const moleculeId_t _id);

signals:
  void atomcountChanged();
  void formulaChanged();
  void indexChanged();
  void nameChanged();
  void tesselationhullChanged();
  void boundingboxChanged();
  void atomInserted(const atomId_t);
  void atomRemoved(const atomId_t);

private:

  void activateObserver();
  void deactivateObserver();

private:
  static int updateAtomCount(
      const boost::function<const moleculeId_t ()> &_getMolIndex);
  static molecule::BoundingBoxInfo updateBoundingBox(
      const boost::function<const moleculeId_t ()> &_getMolIndex);
  static std::string updateFormulaString(
      const boost::function<const moleculeId_t ()> &_getMolIndex);
  static moleculeId_t updateIndex();
  static std::string updateName(
      const boost::function<const moleculeId_t ()> &_getMolIndex);

  //!> list of channels when atom count needs to update
  static const Observable::channels_t AtomCountChannels;
  //!> list of channels when bounding box needs to update
  static const Observable::channels_t BoundingBoxChannels;
  //!> list of channels when formula needs to update
  static const Observable::channels_t FormulaStringChannels;
  //!> list of channels when the index needs to update
  static const Observable::channels_t IndexChannels;
  //!> list of channels when the name needs to update
  static const Observable::channels_t NameChannels;

private:
  /** Observed Values **/

  //!> enumeration of observed values to match with entries in ObservedValues
  enum ObservedTypes {
    //!> contains the current molecule's atom count
    AtomCount,
    //!> contains the current molecule's formula
    FormulaString,
    //!> contains the current molecule index
    MolIndex,
    //!> contains the current molecule name
    MolName,
    //!> contains newest version of the bounding box on request
    BoundingBox,
    //!> gives the size of the enumeration
    MAX_ObservedTypes
  };

  /** Initializes all \a _ObservedValues entries.
   *
   * \param _ObservedValues vector of ObservedValue to be filled
   * \param _moid molecule id
   * \param _molref reference to molecule
   * \param _subjectKilled ref to function to call on subjectKilled()
   */
  static void initObservedValues(
      ObservedValues_t &_ObservedValues,
      const moleculeId_t _molid,
      const molecule * const _molref,
      const boost::function<void(const moleculeId_t)> &_subjectKilled);

  /** Destroys all \a ObservedValues entries.
   *
   * \param _ObservedValues vector of ObservedValue to be destroyed
   */
  static void destroyObservedValues(
      std::vector<boost::any> &_ObservedValues);

  //!> counts how many ObservedValues have already been subjectKilled()
  mutable size_t subjectKilledCount;

private:

  //!> we get multiple subjectKilled(), count and call callback() only on last
  const unsigned int AllsignedOnChannels;
  unsigned int signedOffChannels;

  //!> the Observable we are signed on, also indicates whether we are sign on (not NULL)
  const Observable *owner;

private:

  //!> reference to InstanceBoard for callbacks on subjectKilled()
  QtObservedInstanceBoard & board;

  //!> internal reference to ObservedValues held by QtObservedInstanceBoard
  ObservedValues_t ObservedValues;
};


#endif /* QTOBSERVEDMOLECULE_HPP_ */
