/*
 * VMGJob.hpp
 *
 *  Created on: Jul 12, 2012
 *      Author: heber
 */

#ifndef VMGJOB_HPP_
#define VMGJOB_HPP_


// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "boost/serialization/export.hpp"
#include "boost/serialization/vector.hpp"

#include "JobMarket/Jobs/FragmentJob.hpp"
#include "Fragmentation/Summation/SetValues/SamplingGrid.hpp"
#include "Fragmentation/Summation/Containers/VMGData.hpp"

#include <vector>

/** This class encapsulates a VMG Job.
 *
 *  VMGJob calculates the long-range contribution that is missed out so far in the
 *  MPQCJob where the SCF cycle is calculated. To this end
 *
 */
class VMGJob : public FragmentJob
{
public:
  /** Constructor for class VMGJob.
   *
   * @param _JobId id of the job
   * @param _density_grid sampled electron charge density from short-range solutions
   * @param _particle_positions position per nuclei
   * @param _particle_charges charges per nuclei
   * @param _near_field_cells number of grid-points used to smear our nuclei charge
   * @param _interpolation_degree degree of interpolation polynomial for getting nuclei
   *        potential from grid
   * @param _DoImportParticles whether we import particles (true) or evaluate only
   * @param _DoPrintDebug whether we do print grid for debug visualization or not
   * @param _OpenBoundaryConditions whether we have open (true) boundary conditions or periodic (false)
   * @param _DoSmearCharges whether to smear out electronic charge distributions with bsplines or not
   */
  VMGJob(const JobId_t _JobId,
      const SamplingGrid &density_grid,
      const std::vector< std::vector< double > > &_particle_positions,
      const std::vector< double > &_particle_charges,
      const size_t _near_field_cells,
      const size_t _interpolation_degree,
      const bool _DoImportParticles=true,
      const bool _DoPrintDebug=false,
      const bool _OpenBoundaryConditions=false,
      const bool _DoSmearCharges=false
      );
  virtual ~VMGJob();

  FragmentResult::ptr Work();

private:
  void InitVMG();

  void InitVMGArrays();

private:
  //!> sampled density required as input
  const SamplingGrid density_grid;
  //!> positions of all nuclei
  const std::vector< std::vector< double > > particle_positions;
  //!> charges of all nuclei
  const std::vector< double > particle_charges;
  //!> near field cells used in smearing out core charge density
  const size_t near_field_cells;
  //!> interpolation degree used in sampling the potential of the nuclei
  const size_t interpolation_degree;
  //!> whether we import particles (true) or evaluate only
  const bool DoImportParticles;
  //!> whether we do print grid for debug visualization or not
  const bool DoPrintDebug;
  //!> whether we have open (true) boundary conditions or periodic (false)
  const bool OpenBoundaryConditions;
  //!> whether to smear out electronic charge distributions with bsplines or not
  const bool DoSmearCharges;

private:
  //!> temporary instance to hold return data
  VMGData returndata;

  /** This structure stores particle values per particle.
   *
   * \note This structure contains temporary information needed during solving
   * with VMG.
   * \warning It is specifically not serialized!
   *
   */
  struct particle_arrays {
    particle_arrays();
    ~particle_arrays();

    /** Initializes arrays.
     *
     * @param _num_particles size of array per dimension
     */
    void init(const size_t _num_particles);

    //!> number of particles
    size_t num_particles;
    //!> forces array
    double *f;
    //!> position array
    double *x;
    //!> potential array
    double *p;
    //!> charge array
    double *q;
  } particles;

private:
  /** private default cstor for serialization only
   */
  VMGJob();

  friend class boost::serialization::access;
  // serialization
  template <typename Archive>
  void serialize(Archive& ar, const unsigned int version)
  {
    ar & boost::serialization::base_object<FragmentJob>(*this);
    ar & const_cast< SamplingGrid &>(density_grid);
    ar & const_cast< std::vector< std::vector< double > > &>(particle_positions);
    ar & const_cast< std::vector< double > &>(particle_charges);
    ar & const_cast< size_t &>(near_field_cells);
    ar & const_cast< size_t &>(interpolation_degree);
    ar & const_cast< bool &>(DoImportParticles);
    ar & const_cast< bool &>(DoPrintDebug);
    ar & const_cast< bool &>(OpenBoundaryConditions);
    ar & const_cast< bool &>(DoSmearCharges);
    ar & returndata;
  }
};

// we need to give this class a unique key for serialization
// its is only serialized through its base class FragmentJob
BOOST_CLASS_EXPORT_KEY(VMGJob)

#endif /* VMGJOB_HPP_ */
