/*
 * Project: MoleCuilder
 * Description: creates and alters molecular systems
 * Copyright (C)  2012 University of Bonn. All rights reserved.
 * 
 *
 *   This file is part of MoleCuilder.
 *
 *    MoleCuilder is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    MoleCuilder is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with MoleCuilder.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * HomologyContainerUnitTest.cpp
 *
 *  Created on: Sep 22, 2012
 *      Author: heber
 */

// include config.h
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

using namespace std;

#include <cppunit/CompilerOutputter.h>
#include <cppunit/extensions/TestFactoryRegistry.h>
#include <cppunit/ui/text/TestRunner.h>

// include headers that implement a archive in simple text format
#include <boost/archive/text_oarchive.hpp>
#include <boost/archive/text_iarchive.hpp>

#include <boost/assign.hpp>

#include "Fragmentation/Homology/HomologyContainer.hpp"
#include "Fragmentation/Graph.hpp"

#include "HomologyContainerUnitTest.hpp"

#include <sstream>

using namespace boost::assign;

#ifdef HAVE_TESTRUNNER
#include "UnitTestMain.hpp"
#endif /*HAVE_TESTRUNNER*/

/********************************************** Test classes **************************************/

// Registers the fixture into the 'registry'
CPPUNIT_TEST_SUITE_REGISTRATION( HomologyContainerTest );


void HomologyContainerTest::setUp()
{
  // add nodes
  nodes +=
      FragmentNode(1,1),
      FragmentNode(1,4),
      FragmentNode(2,2),
      FragmentNode(2,4);
  othernodes +=
      FragmentNode(1,1),
      FragmentNode(1,4);

  // add edges
  edges +=
      FragmentEdge(1,1),
      FragmentEdge(1,4),
      FragmentEdge(2,2),
      FragmentEdge(2,4);
  otheredges +=
      FragmentEdge(1,4),
      FragmentEdge(2,2);

  // construct graphs
  HomologyGraph graph(nodes, edges);
  HomologyGraph othergraph(othernodes, otheredges);

  // place in container
  Fragment::position_t pos(3, 0.);
  Fragment::positions_t positions(1, pos);
  Fragment::charges_t charges(1,1.);
  Fragment dummy(positions, charges);
  charges[0] = 6.;
  positions[0][0] = 1.;
  Fragment dummy1(positions, charges);
  positions[0][0] = 2.;
  Fragment dummy2(positions, charges);
  container +=
      std::make_pair( graph, std::make_pair(dummy1, 1.) ),
      std::make_pair( graph, std::make_pair(dummy2, 1.5) ),
      std::make_pair( othergraph, std::make_pair(dummy, 2.) );
  // create HomologyContainer
  Keys = new HomologyContainer(container);
}


void HomologyContainerTest::tearDown()
{
  delete Keys;
}

/** UnitTest for whether homologies are correctly recognized
 */
void HomologyContainerTest::InsertionTest()
{
  // construct graphs
  HomologyGraph graph(nodes, edges);

  HomologyContainer::container_t newcontainer;
  Fragment::position_t pos(3, 0.);
  Fragment::positions_t positions(1, pos);
  Fragment::charges_t charges(1,1.);
  Fragment dummy(positions, charges);
  newcontainer +=
      std::make_pair( graph, std::make_pair(dummy, 1.) );

  Keys->insert(newcontainer);

  CPPUNIT_ASSERT_EQUAL( (size_t)4, Keys->container.size() );
}

/** UnitTest for operator==() works correctly.
 */
void HomologyContainerTest::EqualityTest()
{
  // compare same container
  CPPUNIT_ASSERT( *Keys == *Keys );

  // construct other container
  HomologyGraph graph(nodes, edges);
  HomologyContainer::container_t newcontainer;
  Fragment::position_t pos(3, 0.);
  Fragment::positions_t positions(1, pos);
  Fragment::charges_t charges(1,1.);
  Fragment dummy(positions, charges);
  newcontainer +=
      std::make_pair( graph, std::make_pair(dummy, 1.) );

  HomologyContainer other(newcontainer);

  CPPUNIT_ASSERT( *Keys != other );
}

/** UnitTest for serialization
 */
void HomologyContainerTest::serializeTest()
{
  // serialize
  std::stringstream outputstream;
  boost::archive::text_oarchive oa(outputstream);
  oa << Keys;

  // deserialize
  HomologyContainer *samekeys = NULL;
  std::stringstream returnstream(outputstream.str());
  boost::archive::text_iarchive ia(returnstream);
  ia >> samekeys;

  CPPUNIT_ASSERT( samekeys != NULL );
  //std::cout << *Keys << std::endl;
  CPPUNIT_ASSERT_EQUAL( *Keys, *samekeys );

  delete samekeys;
}
