/*
 * MemDebug.hpp
 *
 *  Created on: Apr 28, 2010
 *      Author: crueger
 */

#ifndef MEMDEBUG_HPP_
#define MEMDEBUG_HPP_

/**
 * @file
 * This module allows easy automatic memory tracking. Link this module to replace the
 * operators new, new[], delete and delete[] with custom versions that add tracking
 * information to allocated blocks.
 *
 * All tracking is done in O(1) for new and delete operators. Summaries for the
 * used memory take O(N), where N is the number of currently allocated memory chunks.
 *
 * To use full tracking including file name and line number include this file in
 * your sourcefiles.
 */
#ifndef NDEBUG
#ifndef NO_MEMDEBUG

#ifndef MEMDEBUG
#define MEMDEBUG
#endif

#include <new>

// some light header files, that do weird new stuff and therefore need
// to be loaded before the define
#include <string>
#include <boost/shared_ptr.hpp>
#include <boost/function.hpp>
#include <boost/program_options.hpp>


namespace Memory {

  /**
   * Displays a short summary of the memory state.
   */
  void getState();

  void _ignore(void*);

  /**
   * Use this to disable memory for a certain pointer on the heap.
   * This is useful for pointers which should live over the run of the
   * program, and which are deleted automatically at the end. Usually these
   * pointers should be wrapped inside some kind of smart pointer to
   * ensure destruction at the end.
   */
  template <typename T>
  T *ignore(T* ptr){
    _ignore((void*)ptr);
    return ptr;
  }
}

void *operator new   (size_t nbytes,const char* file, int line) throw(std::bad_alloc);
void *operator new[] (size_t nbytes,const char* file, int line) throw(std::bad_alloc);
void operator delete   (void *ptr,const char*, int) throw();
void operator delete[] (void *ptr,const char*, int) throw();

/**
 * This macro replaces all occurences of the keyword new with a replaced
 * version that allows tracking.
 */
#define new new(__FILE__,__LINE__)

#endif
#endif


#ifdef NDEBUG
#undef MEMDEBUG
#endif

#ifndef MEMDEBUG
// memory debugging was disabled

namespace Memory {
  inline void getState(){}

  template <typename T>
  inline T *ignore(T* ptr){
    return ptr;
  }
}

#endif
#endif /* MEMDEBUG_HPP_ */
