/*
 * TextDialog.cpp
 *
 *  Created on: Jan 5, 2010
 *      Author: crueger
 */

#include <iostream>

#include "UIElements/TextDialog.hpp"

#include "World.hpp"
#include "periodentafel.hpp"
#include "atom.hpp"
#include "molecule.hpp"
#include "log.hpp"
#include "verbose.hpp"

using namespace std;


TextDialog::TextDialog()
{
}

TextDialog::~TextDialog()
{
}


void TextDialog::queryInt(const char* title, int* target){
  registerQuery(new IntTextQuery(title,target));
}

void TextDialog::queryDouble(const char* title, double* target){
  registerQuery(new DoubleTextQuery(title,target));
}

void TextDialog::queryString(const char* title, string* target){
  registerQuery(new StringTextQuery(title,target));
}

void TextDialog::queryMolecule(const char* title, molecule **target, MoleculeListClass *molecules) {
  registerQuery(new MoleculeTextQuery(title,target,molecules));
}

void TextDialog::queryVector(const char* title, Vector *target,const double *const cellSize, bool check) {
  registerQuery(new VectorTextQuery(title,target,cellSize,check));
}

void TextDialog::queryElement(const char* title, const element **target){
  registerQuery(new ElementTextQuery(title,target));
}

/************************** Query Infrastructure ************************/

TextDialog::IntTextQuery::IntTextQuery(string title,int *_target) :
    Dialog::IntQuery(title,_target)
{}

TextDialog::IntTextQuery::~IntTextQuery() {}

bool TextDialog::IntTextQuery::handle() {
  bool badInput = false;
  do{
    badInput = false;
    Log() << Verbose(0) << getTitle();
    cin >> tmp;
    if(cin.fail()){
      badInput=true;
      cin.clear();
      cin.ignore(std::numeric_limits<streamsize>::max(),'\n');
      Log() << Verbose(0) << "Input was not a number!" << endl;
    }
  } while(badInput);
  // clear the input buffer of anything still in the line
  cin.ignore(std::numeric_limits<streamsize>::max(),'\n');
  return true;
}

TextDialog::StringTextQuery::StringTextQuery(string title,string *_target) :
    Dialog::StringQuery(title,_target)
{}

TextDialog::StringTextQuery::~StringTextQuery() {}

bool TextDialog::StringTextQuery::handle() {
  Log() << Verbose(0) << getTitle();
  getline(cin,tmp);
  return true;
}

TextDialog::DoubleTextQuery::DoubleTextQuery(string title,double *_target) :
    Dialog::DoubleQuery(title,_target)
{}

TextDialog::DoubleTextQuery::~DoubleTextQuery() {}

bool TextDialog::DoubleTextQuery::handle() {
  bool badInput = false;
  do{
    badInput = false;
    Log() << Verbose(0) << getTitle();
    cin >> tmp;
    if(cin.fail()){
      badInput = true;
      cin.clear();
      cin.ignore(std::numeric_limits<streamsize>::max(),'\n');
      Log() << Verbose(0) << "Input was not a number!" << endl;
    }
  }while(badInput);
  cin.ignore(std::numeric_limits<streamsize>::max(),'\n');
  return true;
}

TextDialog::MoleculeTextQuery::MoleculeTextQuery(string title, molecule **_target, MoleculeListClass *_molecules) :
    Dialog::MoleculeQuery(title,_target,_molecules)
{}

TextDialog::MoleculeTextQuery::~MoleculeTextQuery() {}

bool TextDialog::MoleculeTextQuery::handle() {
  int idxOfMol=0;
  bool badInput = false;
  do{
    badInput = false;
    Log() << Verbose(0) << getTitle();
    cin >> idxOfMol;
    if(cin.fail()){
      badInput = true;
      cin.clear();
      cin.ignore(std::numeric_limits<streamsize>::max(),'\n');
      Log() << Verbose(0) << "Input was not a number!" << endl;
      continue;
    }

    tmp = molecules->ReturnIndex(idxOfMol);
    if(!tmp && idxOfMol!=-1){
      Log() << Verbose(0) << "Invalid Molecule Index" << endl;
      badInput = true;
    }

  } while(badInput);
  cin.ignore(std::numeric_limits<streamsize>::max(),'\n');
  return (idxOfMol!=-1);
}

TextDialog::VectorTextQuery::VectorTextQuery(std::string title, Vector *_target, const double *const _cellSize, bool _check) :
    Dialog::VectorQuery(title,_target,_cellSize,_check)
{}

TextDialog::VectorTextQuery::~VectorTextQuery()
{}

bool TextDialog::VectorTextQuery::handle() {
  Log() << Verbose(0) << getTitle();

  char coords[3] = {'x','y','z'};
  int j = -1;
  for (int i=0;i<3;i++) {
    j += i+1;
    do {
      Log() << Verbose(0) << coords[i] << "[0.." << cellSize[j] << "]: ";
      cin >> (*tmp)[i];
    } while ((((*tmp)[i] < 0) || ((*tmp)[i] >= cellSize[j])) && (check));
  }
  return true;
}


TextDialog::ElementTextQuery::ElementTextQuery(std::string title, const element **target) :
    Dialog::ElementQuery(title,target)
{}

TextDialog::ElementTextQuery::~ElementTextQuery()
{}

bool TextDialog::ElementTextQuery::handle() {
  bool badInput=false;
  bool aborted = false;
  do{
    badInput = false;
    Log() << Verbose(0) << getTitle();

    // try to read as Atomic number
    int Z;
    cin >> Z;
    if(!cin.fail()){
      if(Z==-1){
        aborted = true;
      }
      else{
        tmp = World::getInstance().getPeriode()->FindElement(Z);
        if(!tmp){
          Log() << Verbose(0) << "No element with this atomic number!" << endl;
          badInput = true;
        }
      }
      continue;
    }
    else{
      cin.clear();
    }

    // Try to read as shorthand
    // the last buffer content was not removed, so we read the
    // same thing again, this time as a string
    string shorthand;
    cin >> shorthand;
    if(!cin.fail()){
      if(shorthand.empty()){
        aborted = true;
      }
      else{
        tmp = World::getInstance().getPeriode()->FindElement(shorthand.c_str());
        if(!tmp){
          Log() << Verbose(0) << "No element with this shorthand!" << endl;
          badInput = true;
        }
      }
    }
    else{
      Log() << Verbose(0) << "Could not read input. Try Again." << endl;
      cin.clear();
      cin.ignore(std::numeric_limits<streamsize>::max(),'\n');
      badInput = true;
    }

  }while(badInput);
  cin.ignore(std::numeric_limits<streamsize>::max(),'\n');
  return !aborted;
}
