/*
 * LinkedCellUnitTest.cpp
 *
 *  Created on: Apr 9, 2010
 *      Author: heber
 */

using namespace std;

#include <cppunit/CompilerOutputter.h>
#include <cppunit/extensions/TestFactoryRegistry.h>
#include <cppunit/ui/text/TestRunner.h>

#include <iostream>
#include <stdio.h>
#include <cstring>

#include "atom.hpp"
#include "element.hpp"
#include "linkedcell.hpp"
#include "molecule.hpp"
#include "periodentafel.hpp"
#include "LinkedCellUnitTest.hpp"
#include "World.hpp"

#ifdef HAVE_TESTRUNNER
#include "UnitTestMain.hpp"
#endif /*HAVE_TESTRUNNER*/

/********************************************** Test classes **************************************/

// Registers the fixture into the 'registry'
CPPUNIT_TEST_SUITE_REGISTRATION( LinkedCellTest );


void LinkedCellTest::setUp()
{
  atom *Walker = NULL;

  // init private all pointers to zero
  TestMolecule = NULL;
  hydrogen = NULL;
  tafel = NULL;

  // construct element
  hydrogen = new element;
  hydrogen->Z = 1;
  hydrogen->CovalentRadius = 0.23;
  strcpy(hydrogen->name, "hydrogen");
  strcpy(hydrogen->symbol, "H");

  // construct periodentafel
  tafel = World::getInstance().getPeriode();
  tafel->AddElement(hydrogen);

  // construct molecule (water molecule)
  TestMolecule = World::getInstance().createMolecule();
  for (double x=0.5;x<3;x+=1.)
    for (double y=0.5;y<3;y+=1.)
      for (double z=0.5;z<3;z+=1.) {
        Walker = World::getInstance().createAtom();
        Walker->type = hydrogen;
        Walker->node->Init(x, y, z );
        TestMolecule->AddAtom(Walker);
      }

  // construct linked cell
  LC = new LinkedCell (TestMolecule, 1.);

  // check that TestMolecule was correctly constructed
  CPPUNIT_ASSERT_EQUAL( TestMolecule->AtomCount, 3*3*3 );
  Walker = TestMolecule->start->next;
  CPPUNIT_ASSERT( TestMolecule->end != Walker );
};


void LinkedCellTest::tearDown()
{
  delete(LC);
  World::purgeInstance();
  MemoryUsageObserver::purgeInstance();
};


/** UnitTest for LinkedCell::CheckBounds().
 */
void LinkedCellTest::CheckBoundsTest()
{
  // check for within bounds
  LC->n[0] = LC->n[1] = LC->n[2] = 0;
  CPPUNIT_ASSERT_EQUAL( true, LC->CheckBounds() );
  LC->n[0] = LC->n[1] = LC->n[2] = 1;
  CPPUNIT_ASSERT_EQUAL( true, LC->CheckBounds() );
  LC->n[0] = LC->n[1] = LC->n[2] = 2;
  CPPUNIT_ASSERT_EQUAL( true, LC->CheckBounds() );

  // check for out of bounds
  cout << "The following test is supposed to fail and produce an ERROR." << endl;
  LC->n[0] = 404040;
  CPPUNIT_ASSERT_EQUAL( false, LC->CheckBounds() );
  cout << "The following test is supposed to fail and produce an ERROR." << endl;
  LC->n[0] = 0;
  LC->n[1] = 5000;
  CPPUNIT_ASSERT_EQUAL( false, LC->CheckBounds() );
  cout << "The following test is supposed to fail and produce an ERROR." << endl;
  LC->n[1] = 0;
  LC->n[2] = -70;
  CPPUNIT_ASSERT_EQUAL( false, LC->CheckBounds() );
  cout << "The following test is supposed to fail and produce an ERROR." << endl;
  LC->n[0] = LC->n[1] = LC->n[2] = 3;
  CPPUNIT_ASSERT_EQUAL( false, LC->CheckBounds() );
};


/** UnitTest for LinkedCell::GetCurrentCell().
 * Note that CheckBounds() is used and has to be tested already.
 */
void LinkedCellTest::GetCurrentCellTest()
{
  // within bounds
  LC->n[0] = LC->n[1] = LC->n[2] = 0;
  CPPUNIT_ASSERT_EQUAL( (const LinkedCell::LinkedNodes*)&LC->LC[0 * 3*3 + 0 * 3 + 0], LC->GetCurrentCell() );
  LC->n[0] = LC->n[1] = LC->n[2] = 1;
  CPPUNIT_ASSERT_EQUAL( (const LinkedCell::LinkedNodes*)&LC->LC[1 * 3*3 + 1 * 3 + 1], LC->GetCurrentCell() );
  LC->n[0] = LC->n[1] = LC->n[2] = 2;
  CPPUNIT_ASSERT_EQUAL( (const LinkedCell::LinkedNodes*)&LC->LC[2 * 3*3 + 2 * 3 + 2], LC->GetCurrentCell() );

  // out of bounds
  LC->n[0] = LC->n[1] = LC->n[2] = 3;
  cout << "The following test is supposed to fail and produce an ERROR." << endl;
  CPPUNIT_ASSERT_EQUAL( (const LinkedCell::LinkedNodes*)NULL, LC->GetCurrentCell() );
  LC->n[0] = LC->n[1] = LC->n[2] = -1;
  cout << "The following test is supposed to fail and produce an ERROR." << endl;
  CPPUNIT_ASSERT_EQUAL( (const LinkedCell::LinkedNodes*)NULL, LC->GetCurrentCell() );
};

/** UnitTest for LinkedCell::GetRelativeToCurrentCell().
 */
void LinkedCellTest::GetRelativeToCurrentCellTest()
{
  int offset[3];

  // offset to (0,0,0) always
  offset[0] = offset[1] = offset[2] = 0;
  LC->n[0] = LC->n[1] = LC->n[2] = 0;
  CPPUNIT_ASSERT_EQUAL( (const LinkedCell::LinkedNodes*)&LC->LC[0], LC->GetRelativeToCurrentCell(offset) );
  offset[0] = offset[1] = offset[2] = -1;
  LC->n[0] = LC->n[1] = LC->n[2] = 1;
  CPPUNIT_ASSERT_EQUAL( (const LinkedCell::LinkedNodes*)&LC->LC[0], LC->GetRelativeToCurrentCell(offset) );
  offset[0] = offset[1] = offset[2] = -2;
  LC->n[0] = LC->n[1] = LC->n[2] = 2;
  CPPUNIT_ASSERT_EQUAL( (const LinkedCell::LinkedNodes*)&LC->LC[0], LC->GetRelativeToCurrentCell(offset) );

  // offset to (0,0,0) - 1.*(x/y/z) out of bounds
  offset[0] = offset[1] = offset[2] = 0;
  offset[0] = -1;
  LC->n[0] = LC->n[1] = LC->n[2] = 0;
  cout << "The following test is supposed to fail and produce an ERROR." << endl;
  CPPUNIT_ASSERT_EQUAL( (const LinkedCell::LinkedNodes*)NULL, LC->GetRelativeToCurrentCell(offset) );
  offset[0] = offset[1] = offset[2] = 0;
  offset[1] = -1;
  LC->n[0] = LC->n[1] = LC->n[2] = 0;
  cout << "The following test is supposed to fail and produce an ERROR." << endl;
  CPPUNIT_ASSERT_EQUAL( (const LinkedCell::LinkedNodes*)NULL, LC->GetRelativeToCurrentCell(offset) );
  offset[0] = offset[1] = offset[2] = 0;
  offset[2] = -1;
  LC->n[0] = LC->n[1] = LC->n[2] = 0;
  cout << "The following test is supposed to fail and produce an ERROR." << endl;
  CPPUNIT_ASSERT_EQUAL( (const LinkedCell::LinkedNodes*)NULL, LC->GetRelativeToCurrentCell(offset) );

  // out of bounds
  offset[0] = offset[1] = offset[2] = -5054932;
  LC->n[0] = LC->n[1] = LC->n[2] = 1;
  cout << "The following test is supposed to fail and produce an ERROR." << endl;
  CPPUNIT_ASSERT_EQUAL( (const LinkedCell::LinkedNodes*)NULL, LC->GetRelativeToCurrentCell(offset) );
  offset[0] = offset[1] = offset[2] = 192345;
  LC->n[0] = LC->n[1] = LC->n[2] = 1;
  cout << "The following test is supposed to fail and produce an ERROR." << endl;
  CPPUNIT_ASSERT_EQUAL( (const LinkedCell::LinkedNodes*)NULL, LC->GetRelativeToCurrentCell(offset) );

  // index is out of bounds, offset points within
  offset[0] = offset[1] = offset[2] = -2;
  LC->n[0] = LC->n[1] = LC->n[2] = 4;
  CPPUNIT_ASSERT_EQUAL( (const LinkedCell::LinkedNodes*)&LC->LC[2 * 3*3 + 2 * 3 + 2], LC->GetRelativeToCurrentCell(offset) );

  // index is within bounds, offset points out
  offset[0] = offset[1] = offset[2] = 2;
  LC->n[0] = LC->n[1] = LC->n[2] = 2;
  cout << "The following test is supposed to fail and produce an ERROR." << endl;
  CPPUNIT_ASSERT_EQUAL( (const LinkedCell::LinkedNodes*)NULL, LC->GetRelativeToCurrentCell(offset) );
};


/** UnitTest for LinkedCell::SetIndexToNode().
 */
void LinkedCellTest::SetIndexToNodeTest()
{
  // check all atoms
  atom *Walker = TestMolecule->start;
  while (Walker->next != TestMolecule->end) {
    Walker = Walker->next;
    CPPUNIT_ASSERT_EQUAL( true, LC->SetIndexToNode(Walker) );
  }

  // check internal vectors, returns false, because this atom is not in LC-list!
  Walker = World::getInstance().createAtom();
  Walker->Name = Malloc<char>(6, "LinkedCellTest::SetIndexToNodeTest - Walker");
  strcpy(Walker->Name, "test");
  Walker->x.Init(1,1,1);
  CPPUNIT_ASSERT_EQUAL( false, LC->SetIndexToNode(Walker) );
  World::getInstance().destroyAtom(Walker);

  // check out of bounds vectors
  Walker = World::getInstance().createAtom();
  Walker->Name = Malloc<char>(6, "LinkedCellTest::SetIndexToNodeTest - Walker");
  strcpy(Walker->Name, "test");
  Walker->x.Init(0,-1,0);
  CPPUNIT_ASSERT_EQUAL( false, LC->SetIndexToNode(Walker) );
  World::getInstance().destroyAtom(Walker);
};


/** UnitTest for LinkedCell::SetIndexToVector().
 */
void LinkedCellTest::SetIndexToVectorTest()
{
  Vector tester;

  // check center of each cell
  for (double x=0.5;x<3;x+=1.)
    for (double y=0.5;y<3;y+=1.)
      for (double z=0.5;z<3;z+=1.) {
        tester.Init(x,y,z);
        CPPUNIT_ASSERT_EQUAL( true, LC->SetIndexToVector(&tester) );
      }
  // check corners of each cell
  for (double x=1.;x<4;x+=1.)
    for (double y=1.;y<4;y+=1.)
      for (double z=1.;z<4;z+=1.) {
        tester.Init(x,y,z);
        cout << "Tester is at " << tester << "." << endl;
        CPPUNIT_ASSERT_EQUAL( true, LC->SetIndexToVector(&tester) );
      }
  // check out of bounds
  for (double x=0.5-1e-10;x<5;x+=3.1)
    for (double y=0.5-1e-10;y<5;y+=3.1)
      for (double z=0.5-1e-10;z<5;z+=3.1) {
        tester.Init(x,y,z);
        cout << "The following test is supposed to fail and produce an ERROR." << endl;
        CPPUNIT_ASSERT_EQUAL( false, LC->SetIndexToVector(&tester) );
      }
  // check nonsense vectors
  tester.Init(-423598,3245978,29349);
  cout << "The following test is supposed to fail and produce an ERROR." << endl;
  CPPUNIT_ASSERT_EQUAL( false, LC->SetIndexToVector(&tester) );
};


/** UnitTest for LinkedCell::GetNeighbourBounds().
 */
void LinkedCellTest::GetNeighbourBoundsTest()
{
  Vector tester;
  int lower[NDIM], upper[NDIM];

  tester.Init(0.5,0.5,0.5);
  LC->SetIndexToVector(&tester);
  LC->GetNeighbourBounds(lower, upper);
  for (int i=0;i<NDIM;i++)
    CPPUNIT_ASSERT_EQUAL( 0, lower[i]);
  for (int i=0;i<NDIM;i++)
    CPPUNIT_ASSERT_EQUAL( 1, upper[i]);
};


/** UnitTest for LinkedCell::GetallNeighbours().
 */
void LinkedCellTest::GetallNeighboursTest()
{
  Vector tester;
  LinkedCell::LinkedNodes *ListOfPoints = NULL;
  atom *Walker = NULL;
  size_t size = 0;

  // get all atoms
  tester.Init(1.5,1.5,1.5);
  CPPUNIT_ASSERT_EQUAL ( true, LC->SetIndexToVector(&tester) );
  ListOfPoints = LC->GetallNeighbours();
  size = ListOfPoints->size();
  CPPUNIT_ASSERT_EQUAL( (size_t)27, size );
  Walker = TestMolecule->start;
  Walker = TestMolecule->start;
  while (Walker->next != TestMolecule->end) {
    Walker = Walker->next;
    ListOfPoints->remove(Walker);
    size--;
    CPPUNIT_ASSERT_EQUAL( size, ListOfPoints->size() );
  }
  CPPUNIT_ASSERT_EQUAL( (size_t)0, size );
  CPPUNIT_ASSERT_EQUAL( (size_t)0, ListOfPoints->size() );
  CPPUNIT_ASSERT_EQUAL( true, ListOfPoints->empty() );
  delete(ListOfPoints);

  // get all atoms in one corner
  tester.Init(0.5, 0.5, 0.5);
  CPPUNIT_ASSERT_EQUAL ( true, LC->SetIndexToVector(&tester) );
  ListOfPoints = LC->GetallNeighbours();
  size=ListOfPoints->size();
  CPPUNIT_ASSERT_EQUAL( (size_t)8, size );
  Walker = TestMolecule->start;
  while (Walker->next != TestMolecule->end) {
    Walker = Walker->next;
    if ((Walker->x.x[0] <2) && (Walker->x.x[1] <2) && (Walker->x.x[2] <2)) {
      ListOfPoints->remove(Walker);
      size--;
      CPPUNIT_ASSERT_EQUAL( size, ListOfPoints->size() );
    }
  }
  CPPUNIT_ASSERT_EQUAL( (size_t)0, size );
  CPPUNIT_ASSERT_EQUAL( (size_t)0, ListOfPoints->size() );
  CPPUNIT_ASSERT_EQUAL( true, ListOfPoints->empty() );
  delete(ListOfPoints);

  // get all atoms from one corner
  tester.Init(0.5, 0.5, 0.5);
  CPPUNIT_ASSERT_EQUAL ( true, LC->SetIndexToVector(&tester) );
  ListOfPoints = LC->GetallNeighbours(3);
  size=ListOfPoints->size();
  CPPUNIT_ASSERT_EQUAL( (size_t)27, size );
  Walker = TestMolecule->start;
  while (Walker->next != TestMolecule->end) {
    Walker = Walker->next;
    ListOfPoints->remove(Walker);
    size--;
    CPPUNIT_ASSERT_EQUAL( size, ListOfPoints->size() );
  }
  CPPUNIT_ASSERT_EQUAL( (size_t)0, size );
  CPPUNIT_ASSERT_EQUAL( (size_t)0, ListOfPoints->size() );
  CPPUNIT_ASSERT_EQUAL( true, ListOfPoints->empty() );
  delete(ListOfPoints);
};


/** UnitTest for LinkedCell::GetPointsInsideSphere().
 */
void LinkedCellTest::GetPointsInsideSphereTest()
{
  Vector tester;
  LinkedCell::LinkedNodes *ListOfPoints = NULL;
  atom *Walker = NULL;
  size_t size = 0;

  // get all points around central arom with radius 1.
  tester.Init(1.5,1.5,1.5);
  CPPUNIT_ASSERT_EQUAL ( true, LC->SetIndexToVector(&tester) );
  ListOfPoints = LC->GetPointsInsideSphere(1., &tester);
  size = ListOfPoints->size();
  CPPUNIT_ASSERT_EQUAL( (size_t)7, size );
  Walker = TestMolecule->start;
  while (Walker->next != TestMolecule->end) {
    Walker = Walker->next;
    if ((Walker->x.DistanceSquared(&tester) - 1.) < MYEPSILON ) {
      ListOfPoints->remove(Walker);
      size--;
      CPPUNIT_ASSERT_EQUAL( size, ListOfPoints->size() );
    }
  }
  CPPUNIT_ASSERT_EQUAL( (size_t)0, size );
  CPPUNIT_ASSERT_EQUAL( (size_t)0, ListOfPoints->size() );
  CPPUNIT_ASSERT_EQUAL( true, ListOfPoints->empty() );
  delete(ListOfPoints);
};
