/*
 * AnalysisCorrelationToSurfaceUnitTest.cpp
 *
 *  Created on: Oct 13, 2009
 *      Author: heber
 */

using namespace std;

#include <cppunit/CompilerOutputter.h>
#include <cppunit/extensions/TestFactoryRegistry.h>
#include <cppunit/ui/text/TestRunner.h>

#include <cstring>

#include "analysis_correlation.hpp"
#include "AnalysisCorrelationToSurfaceUnitTest.hpp"

#include "World.hpp"
#include "atom.hpp"
#include "boundary.hpp"
#include "element.hpp"
#include "molecule.hpp"
#include "linkedcell.hpp"
#include "periodentafel.hpp"
#include "tesselation.hpp"
#include "World.hpp"

#include "Helpers/Assert.hpp"

#ifdef HAVE_TESTRUNNER
#include "UnitTestMain.hpp"
#endif /*HAVE_TESTRUNNER*/

/********************************************** Test classes **************************************/

// Registers the fixture into the 'registry'
CPPUNIT_TEST_SUITE_REGISTRATION( AnalysisCorrelationToSurfaceUnitTest );

void AnalysisCorrelationToSurfaceUnitTest::setUp()
{
  ASSERT_DO(Assert::Throw);

  atom *Walker = NULL;

  // init private all pointers to zero
  TestList = NULL;
  TestMolecule = NULL;
  hydrogen = NULL;
  tafel = NULL;
  surfacemap = NULL;
  binmap = NULL;
  Surface = NULL;
  LC = NULL;

  // construct element
  hydrogen = new element;
  hydrogen->Z = 1;
  strcpy(hydrogen->name, "hydrogen");
  strcpy(hydrogen->symbol, "H");
  carbon = new element;
  carbon->Z = 6;
  strcpy(carbon->name, "carbon");
  strcpy(carbon->symbol, "C");

  // construct periodentafel
  tafel = World::getInstance().getPeriode();
  tafel->AddElement(hydrogen);
  tafel->AddElement(carbon);

  // construct molecule (tetraeder of hydrogens) base
  TestSurfaceMolecule = World::getInstance().createMolecule();
  Walker = World::getInstance().createAtom();
  Walker->type = hydrogen;
  Walker->node->Init(1., 0., 1. );
  TestSurfaceMolecule->AddAtom(Walker);
  Walker = World::getInstance().createAtom();
  Walker->type = hydrogen;
  Walker->node->Init(0., 1., 1. );
  TestSurfaceMolecule->AddAtom(Walker);
  Walker = World::getInstance().createAtom();
  Walker->type = hydrogen;
  Walker->node->Init(1., 1., 0. );
  TestSurfaceMolecule->AddAtom(Walker);
  Walker = World::getInstance().createAtom();
  Walker->type = hydrogen;
  Walker->node->Init(0., 0., 0. );
  TestSurfaceMolecule->AddAtom(Walker);

  // check that TestMolecule was correctly constructed
  CPPUNIT_ASSERT_EQUAL( TestSurfaceMolecule->AtomCount, 4 );

  TestList = World::getInstance().getMolecules();
  TestSurfaceMolecule->ActiveFlag = true;
  TestList->insert(TestSurfaceMolecule);

  // init tesselation and linked cell
  Surface = new Tesselation;
  LC = new LinkedCell(TestSurfaceMolecule, 5.);
  FindNonConvexBorder(TestSurfaceMolecule, Surface, (const LinkedCell *&)LC, 2.5, NULL);

  // add outer atoms
  TestMolecule = World::getInstance().createMolecule();
  Walker = World::getInstance().createAtom();
  Walker->type = carbon;
  Walker->node->Init(4., 0., 4. );
  TestMolecule->AddAtom(Walker);
  Walker = World::getInstance().createAtom();
  Walker->type = carbon;
  Walker->node->Init(0., 4., 4. );
  TestMolecule->AddAtom(Walker);
  Walker = World::getInstance().createAtom();
  Walker->type = carbon;
  Walker->node->Init(4., 4., 0. );
  TestMolecule->AddAtom(Walker);
  // add inner atoms
  Walker = World::getInstance().createAtom();
  Walker->type = carbon;
  Walker->node->Init(0.5, 0.5, 0.5 );
  TestMolecule->AddAtom(Walker);
  TestMolecule->ActiveFlag = true;
  TestList->insert(TestMolecule);

  // init maps
  surfacemap = NULL;
  binmap = NULL;

};


void AnalysisCorrelationToSurfaceUnitTest::tearDown()
{
  if (surfacemap != NULL)
    delete(surfacemap);
  if (binmap != NULL)
    delete(binmap);

  delete(Surface);
  // note that all the atoms are cleaned by TestMolecule
  delete(LC);
  World::purgeInstance();
  MemoryUsageObserver::purgeInstance();
  logger::purgeInstance();
};


/** Checks whether setup() does the right thing.
 */
void AnalysisCorrelationToSurfaceUnitTest::SurfaceTest()
{
  CPPUNIT_ASSERT_EQUAL( 4, TestSurfaceMolecule->AtomCount );
  CPPUNIT_ASSERT_EQUAL( 4, TestMolecule->AtomCount );
  CPPUNIT_ASSERT_EQUAL( (size_t)2, TestList->ListOfMolecules.size() );
  CPPUNIT_ASSERT_EQUAL( (size_t)4, Surface->PointsOnBoundary.size() );
  CPPUNIT_ASSERT_EQUAL( (size_t)6, Surface->LinesOnBoundary.size() );
  CPPUNIT_ASSERT_EQUAL( (size_t)4, Surface->TrianglesOnBoundary.size() );
};

void AnalysisCorrelationToSurfaceUnitTest::CorrelationToSurfaceTest()
{
  // do the pair correlation
  surfacemap = CorrelationToSurface( TestList, hydrogen, Surface, LC );
//  OutputCorrelationToSurface ( (ofstream *)&cout, surfacemap );
  CPPUNIT_ASSERT( surfacemap != NULL );
  CPPUNIT_ASSERT_EQUAL( (size_t)4, surfacemap->size() );
};

void AnalysisCorrelationToSurfaceUnitTest::CorrelationToSurfaceHydrogenBinNoRangeTest()
{
  BinPairMap::iterator tester;
  surfacemap = CorrelationToSurface( TestList, hydrogen, Surface, LC );
  // put pair correlation into bins and check with no range
//  OutputCorrelationToSurface ( (ofstream *)&cout, surfacemap );
  binmap = BinData( surfacemap, 0.5, 0., 0. );
  CPPUNIT_ASSERT_EQUAL( (size_t)1, binmap->size() );
  OutputCorrelation ( (ofstream *)&cout, binmap );
  tester = binmap->begin();
  CPPUNIT_ASSERT_EQUAL( 0., tester->first );
  CPPUNIT_ASSERT_EQUAL( 4, tester->second );

};

void AnalysisCorrelationToSurfaceUnitTest::CorrelationToSurfaceHydrogenBinRangeTest()
{
  BinPairMap::iterator tester;
  surfacemap = CorrelationToSurface( TestList, hydrogen, Surface, LC );
//  OutputCorrelationToSurface ( (ofstream *)&cout, surfacemap );
  // ... and check with [0., 2.] range
  binmap = BinData( surfacemap, 0.5, 0., 2. );
  CPPUNIT_ASSERT_EQUAL( (size_t)5, binmap->size() );
//  OutputCorrelation ( (ofstream *)&cout, binmap );
  tester = binmap->begin();
  CPPUNIT_ASSERT_EQUAL( 0., tester->first );
  CPPUNIT_ASSERT_EQUAL( 4, tester->second );
  tester = binmap->find(1.);
  CPPUNIT_ASSERT_EQUAL( 1., tester->first );
  CPPUNIT_ASSERT_EQUAL( 0, tester->second );

};

void AnalysisCorrelationToSurfaceUnitTest::CorrelationToSurfaceCarbonBinNoRangeTest()
{
  BinPairMap::iterator tester;
  surfacemap = CorrelationToSurface( TestList, carbon, Surface, LC );
//  OutputCorrelationToSurface ( (ofstream *)&cout, surfacemap );
  // put pair correlation into bins and check with no range
  binmap = BinData( surfacemap, 0.5, 0., 0. );
  //OutputCorrelation ( (ofstream *)&cout, binmap );
  CPPUNIT_ASSERT_EQUAL( (size_t)9, binmap->size() );
  // inside point is first and must have negative value
  tester = binmap->lower_bound(4.25-0.5); // start depends on the min value and
  CPPUNIT_ASSERT( tester != binmap->end() );
  CPPUNIT_ASSERT_EQUAL( 3, tester->second );
  // inner point
  tester = binmap->lower_bound(0.);
  CPPUNIT_ASSERT( tester != binmap->end() );
  CPPUNIT_ASSERT_EQUAL( 1, tester->second );
};

void AnalysisCorrelationToSurfaceUnitTest::CorrelationToSurfaceCarbonBinRangeTest()
{
  BinPairMap::iterator tester;
  surfacemap = CorrelationToSurface( TestList, carbon, Surface, LC );
//  OutputCorrelationToSurface ( (ofstream *)&cout, surfacemap );
  // ... and check with [0., 2.] range
  binmap = BinData( surfacemap, 0.5, -2., 4. );
  //OutputCorrelation ( (ofstream *)&cout, binmap );
  CPPUNIT_ASSERT_EQUAL( (size_t)13, binmap->size() );
  // three outside points
  tester = binmap->lower_bound(4.25-0.5);
  CPPUNIT_ASSERT( tester != binmap->end() );
  CPPUNIT_ASSERT_EQUAL( 3, tester->second );
  // inner point
  tester = binmap->lower_bound(0.);
  CPPUNIT_ASSERT( tester != binmap->end() );
  CPPUNIT_ASSERT_EQUAL( 1, tester->second );
};
