/*
 * CommandLineDialog.cpp
 *
 *  Created on: May 8, 2010
 *      Author: heber
 */


#include <iostream>

#include "UIElements/CommandLineDialog.hpp"

#include "World.hpp"
#include "periodentafel.hpp"
#include "atom.hpp"
#include "molecule.hpp"
#include "log.hpp"
#include "verbose.hpp"

using namespace std;


CommandLineDialog::CommandLineDialog()
{
}

CommandLineDialog::~CommandLineDialog()
{
}


void CommandLineDialog::queryInt(const char* title, int* target){
  registerQuery(new IntTextQuery(title,target));
}

void CommandLineDialog::queryDouble(const char* title, double* target){
  registerQuery(new DoubleTextQuery(title,target));
}

void CommandLineDialog::queryString(const char* title, string* target){
  registerQuery(new StringTextQuery(title,target));
}

void CommandLineDialog::queryMolecule(const char* title, molecule **target, MoleculeListClass *molecules) {
  registerQuery(new MoleculeTextQuery(title,target,molecules));
}

void CommandLineDialog::queryVector(const char* title, Vector *target,const double *const cellSize, bool check) {
  registerQuery(new VectorTextQuery(title,target,cellSize,check));
}

void CommandLineDialog::queryElement(const char* title, const element **target){
  registerQuery(new ElementTextQuery(title,target));
}

/************************** Query Infrastructure ************************/

CommandLineDialog::IntTextQuery::IntTextQuery(string title,int *_target) :
    Dialog::IntQuery(title,_target)
{}

CommandLineDialog::IntTextQuery::~IntTextQuery() {}

bool CommandLineDialog::IntTextQuery::handle() {
  bool badInput = false;
  do{
    badInput = false;
    Log() << Verbose(0) << getTitle();
    cin >> tmp;
    if(cin.fail()){
      badInput=true;
      cin.clear();
      cin.ignore(std::numeric_limits<streamsize>::max(),'\n');
      Log() << Verbose(0) << "Input was not a number!" << endl;
    }
  } while(badInput);
  // clear the input buffer of anything still in the line
  cin.ignore(std::numeric_limits<streamsize>::max(),'\n');
  return true;
}

CommandLineDialog::StringTextQuery::StringTextQuery(string title,string *_target) :
    Dialog::StringQuery(title,_target)
{}

CommandLineDialog::StringTextQuery::~StringTextQuery() {}

bool CommandLineDialog::StringTextQuery::handle() {
  Log() << Verbose(0) << getTitle();
  getline(cin,tmp);
  return true;
}

CommandLineDialog::DoubleTextQuery::DoubleTextQuery(string title,double *_target) :
    Dialog::DoubleQuery(title,_target)
{}

CommandLineDialog::DoubleTextQuery::~DoubleTextQuery() {}

bool CommandLineDialog::DoubleTextQuery::handle() {
  bool badInput = false;
  do{
    badInput = false;
    Log() << Verbose(0) << getTitle();
    cin >> tmp;
    if(cin.fail()){
      badInput = true;
      cin.clear();
      cin.ignore(std::numeric_limits<streamsize>::max(),'\n');
      Log() << Verbose(0) << "Input was not a number!" << endl;
    }
  }while(badInput);
  cin.ignore(std::numeric_limits<streamsize>::max(),'\n');
  return true;
}

CommandLineDialog::MoleculeTextQuery::MoleculeTextQuery(string title, molecule **_target, MoleculeListClass *_molecules) :
    Dialog::MoleculeQuery(title,_target,_molecules)
{}

CommandLineDialog::MoleculeTextQuery::~MoleculeTextQuery() {}

bool CommandLineDialog::MoleculeTextQuery::handle() {
  int idxOfMol=0;
  bool badInput = false;
  do{
    badInput = false;
    Log() << Verbose(0) << getTitle();
    cin >> idxOfMol;
    if(cin.fail()){
      badInput = true;
      cin.clear();
      cin.ignore(std::numeric_limits<streamsize>::max(),'\n');
      Log() << Verbose(0) << "Input was not a number!" << endl;
      continue;
    }

    tmp = molecules->ReturnIndex(idxOfMol);
    if(!tmp && idxOfMol!=-1){
      Log() << Verbose(0) << "Invalid Molecule Index" << endl;
      badInput = true;
    }

  } while(badInput);
  cin.ignore(std::numeric_limits<streamsize>::max(),'\n');
  return (idxOfMol!=-1);
}

CommandLineDialog::VectorTextQuery::VectorTextQuery(std::string title, Vector *_target, const double *const _cellSize, bool _check) :
    Dialog::VectorQuery(title,_target,_cellSize,_check)
{}

CommandLineDialog::VectorTextQuery::~VectorTextQuery()
{}

bool CommandLineDialog::VectorTextQuery::handle() {
  Log() << Verbose(0) << getTitle();

  char coords[3] = {'x','y','z'};
  int j = -1;
  for (int i=0;i<3;i++) {
    j += i+1;
    do {
      Log() << Verbose(0) << coords[i] << "[0.." << cellSize[j] << "]: ";
      cin >> (*tmp)[i];
    } while ((((*tmp)[i] < 0) || ((*tmp)[i] >= cellSize[j])) && (check));
  }
  return true;
}


CommandLineDialog::ElementTextQuery::ElementTextQuery(std::string title, const element **target) :
    Dialog::ElementQuery(title,target)
{}

CommandLineDialog::ElementTextQuery::~ElementTextQuery()
{}

bool CommandLineDialog::ElementTextQuery::handle() {
  bool badInput=false;
  bool aborted = false;
  do{
    badInput = false;
    Log() << Verbose(0) << getTitle();

    // try to read as Atomic number
    int Z;
    cin >> Z;
    if(!cin.fail()){
      if(Z==-1){
        aborted = true;
      }
      else{
        tmp = World::getInstance().getPeriode()->FindElement(Z);
        if(!tmp){
          Log() << Verbose(0) << "No element with this atomic number!" << endl;
          badInput = true;
        }
      }
      continue;
    }
    else{
      cin.clear();
    }

    // Try to read as shorthand
    // the last buffer content was not removed, so we read the
    // same thing again, this time as a string
    string shorthand;
    cin >> shorthand;
    if(!cin.fail()){
      if(shorthand.empty()){
        aborted = true;
      }
      else{
        tmp = World::getInstance().getPeriode()->FindElement(shorthand.c_str());
        if(!tmp){
          Log() << Verbose(0) << "No element with this shorthand!" << endl;
          badInput = true;
        }
      }
    }
    else{
      Log() << Verbose(0) << "Could not read input. Try Again." << endl;
      cin.clear();
      cin.ignore(std::numeric_limits<streamsize>::max(),'\n');
      badInput = true;
    }

  }while(badInput);
  cin.ignore(std::numeric_limits<streamsize>::max(),'\n');
  return !aborted;
}
